/*
	WinSTon

	Create Decode/Diassembly lookup tables

	We base this around a string for each instruction. This is similar to the defines
	found in the DevPac pocket guide. Defines can be 'rrr' for register, 'eeeeee' for
	effective address etc... The code recurses around the string replacing these token
	characters with 0 or 1 and stores them into our tables.
	
	We build up a list of instructions and store the decode/disassembly function in two
	tables. Then we can simply do 'call(DecodeTable[OpCode])'!
	So no nasty masking opcode to find which functions to call.
*/

#include "..\includes\winston.h"
#include "..\includes\decode.h"
#include "..\includes\debug.h"
#include "..\includes\debugger.h"
#include "..\includes\disass.h"
#include "..\includes\effaddr.h"
#include "..\includes\gemdos.h"
#include "..\includes\m68000.h"
#include "..\includes\memory.h"
#include "..\includes\misc.h"
#include "..\includes\stmemory.h"
#include "..\includes\tables.h"
#include "..\includes\tos.h"
#include "..\includes\vdi.h"

//-----------------------------------------------------------------------
// List of strings to scan for in instruction defines
char *pszScanStrings[] = {
	"xxx",			// RegX 0,1,2...7
	"yyy",			// RegY 0,1,2...7
	"rrr",			// Reg 0,1,2...7
	"ddd",			// Reg 0,1,2...7
	"cccc",			// Condition code 0,1,2...15
	"CCCC",			// Bcc condition code 2...15(ignore 0 and 1)
	"vvvv",			// Vector 0,1,2...15
	"oooooooo",		// Offset 0,1,2...255
	"ffffffff",		// Offset 1,2...255
	"RRR",			// Effective Address Reg 0,1,2...7
	"iiiiiiiiiiii",	// Line A or F call

	NULL	// Term
};

// Handy ^2 table
int Power2Table[] = { 1,2,4,8, 16,32,64,128, 256,512,1024,2048, 4096,8192,16384,32768 };
int TableIndex;	// Index into our 'TablePopulate' array

//-----------------------------------------------------------------------
unsigned long EffAddrFunctions_Byte[64];		// Effective Address functions
unsigned long EffAddrFunctions_Word[64];
unsigned long EffAddrFunctions_Long[64];
unsigned long LoadEffAddrFunctions[64];			// Load effective address
unsigned long EffAddrFunctions_ByteCycles[64];	// Cycle counts
unsigned long EffAddrFunctions_WordCycles[64];
unsigned long EffAddrFunctions_LongCycles[64];
unsigned long LoadEffAddrFunctions_LongCycles[64];

//-----------------------------------------------------------------------
// Effective Address table functions, used to access Dn or An as <ea> directly!
// NOTE : Byte,Word and Long are all same
unsigned long EffAddrTable_Dn[] = {
	(unsigned long)effaddr_read_000_D0, (unsigned long)effaddr_read_000_D1, (unsigned long)effaddr_read_000_D2, (unsigned long)effaddr_read_000_D3,
	(unsigned long)effaddr_read_000_D4, (unsigned long)effaddr_read_000_D5, (unsigned long)effaddr_read_000_D6, (unsigned long)effaddr_read_000_D7
};

unsigned long EffAddrTable_An[] = {
	(unsigned long)effaddr_read_001_A0, (unsigned long)effaddr_read_001_A1, (unsigned long)effaddr_read_001_A2, (unsigned long)effaddr_read_001_A3,
	(unsigned long)effaddr_read_001_A4, (unsigned long)effaddr_read_001_A5, (unsigned long)effaddr_read_001_A6, (unsigned long)effaddr_read_001_A7
};

//-----------------------------------------------------------------------
// Effective address defines
#define	EFFADDR_0001000_000_00	BIN12(0,0,0,1,0,0,0, 0,0,0, 0,0)
#define	EFFADDR_0010011_110_00	BIN12(0,0,1,0,0,1,1, 1,1,0, 0,0)
#define	EFFADDR_0000100_000_00	BIN12(0,0,0,0,1,0,0, 0,0,0, 0,0)
#define	EFFADDR_1000000_000_00	BIN12(1,0,0,0,0,0,0, 0,0,0, 0,0)
#define EFFADDR_0100000_000_00	BIN12(0,1,0,0,0,0,0, 0,0,0, 0,0)
#define	EFFADDR_1111111_111_11	BIN12(1,1,1,1,1,1,1, 1,1,1, 1,1)
#define	EFFADDR_0111111_111_11	BIN12(0,1,1,1,1,1,1, 1,1,1, 1,1)
#define	EFFADDR_0011111_111_11	BIN12(0,0,1,1,1,1,1, 1,1,1, 1,1)
#define	EFFADDR_0011111_110_00	BIN12(0,0,1,1,1,1,1, 1,1,0, 0,0)
#define	EFFADDR_0011111_110_11	BIN12(0,0,1,1,1,1,1, 1,1,0, 1,1)
#define	EFFADDR_0010011_110_11	BIN12(0,0,1,0,0,1,1, 1,1,0, 1,1)
#define	EFFADDR_0011011_110_11	BIN12(0,0,1,1,0,1,1, 1,1,0, 1,1)
#define	EFFADDR_0011011_110_00	BIN12(0,0,1,1,0,1,1, 1,1,0, 0,0)
#define	EFFADDR_0010111_110_00	BIN12(0,0,1,0,1,1,1, 1,1,0, 0,0)
#define	EFFADDR_1111111_110_00	BIN12(1,1,1,1,1,1,1, 1,1,0, 0,0)
#define	EFFADDR_1011111_111_11	BIN12(1,0,1,1,1,1,1, 1,1,1, 1,1)
#define	EFFADDR_1011111_110_11	BIN12(1,0,1,1,1,1,1, 1,1,0, 1,1)
#define	EFFADDR_1011111_110_00	BIN12(1,0,1,1,1,1,1, 1,1,0, 0,0)
#define	EFFADDR_0010000_000_00	BIN12(0,0,1,0,0,0,0, 0,0,0, 0,0)
#define	EFFADDR_0001000_000_00	BIN12(0,0,0,1,0,0,0, 0,0,0, 0,0)
#define	EFFADDR_0000010_000_00	BIN12(0,0,0,0,0,1,0, 0,0,0, 0,0)
#define	EFFADDR_0000001_000_00	BIN12(0,0,0,0,0,0,1, 0,0,0, 0,0)
#define	EFFADDR_0000000_100_00	BIN12(0,0,0,0,0,0,0, 1,0,0, 0,0)
#define	EFFADDR_0000000_010_00	BIN12(0,0,0,0,0,0,0, 0,1,0, 0,0)
#define	EFFADDR_0000000_000_10	BIN12(0,0,0,0,0,0,0, 0,0,0, 1,0)
#define	EFFADDR_0000000_000_01	BIN12(0,0,0,0,0,0,0, 0,0,0, 0,1)

#ifdef USE_DEBUGGER
	#define	DISFUNC(func)		func
#else
	#define	DISFUNC(func)		NULL
#endif

//-----------------------------------------------------------------------
// Population table, entry for each instruction type
TABLE_POPULATE TablePopulate[] = {
	{ "1100xxx1 00000yyy", NULL,					DISFUNC(Disass_ABCD_Dy_Dx),abcd_dn_dn,OP_CYCLES(6,1,0),TBL_DNUPR_1|TBL_DN_2 },						// ABCD Dy,Dx
	{ "1100xxx1 00001yyy", NULL,					DISFUNC(Disass_ABCD_Ay_Ax),abcd_an_an,OP_CYCLES(18,3,1),TBL_ANUPR_1|TBL_AN_2 },						// ABCD -(Ay),-(Ax)
	{ "1101rrr0 00eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_ADD_ea_Dn),add_ea_dn_byte,OP_CYCLES(4,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// ADD.B <ea>,Dn	(no byte access)
	{ "1101rrr0 01eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_ADD_ea_Dn),add_ea_dn_word,OP_CYCLES(4,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// ADD.W <ea>,Dn
	{ "1101rrr0 10eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_ADD_ea_Dn),add_ea_dn_long,OP_CYCLES(6,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// ADD.L <ea>,Dn
	{ "1101rrr1 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ADD_Dn_ea),add_dn_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// ADD.B Dn,<ea>
	{ "1101rrr1 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ADD_Dn_ea),add_dn_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// ADD.W Dn,<ea>
	{ "1101rrr1 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ADD_Dn_ea),add_dn_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// ADD.L Dn,<ea>
	{ "1101rrr0 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_ADDA),adda_ea_an_word,OP_CYCLES(8,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// ADDA.W <ea>,An
	{ "1101rrr1 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_ADDA),adda_ea_an_long,OP_CYCLES(6,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// ADDA.L <ea>,An
	{ "00000110 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ADDI),addi_imm_dn_byte,OP_CYCLES(8,2,0),TBL_DN_1 },							// ADDI.B #<data>,Dn
	{ "00000110 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ADDI),addi_imm_dn_word,OP_CYCLES(8,2,0),TBL_DN_1 },							// ADDI.W #<data>,Dn
	{ "00000110 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ADDI),addi_imm_dn_long,OP_CYCLES(16,3,0),TBL_DN_1 },							// ADDI.L #<data>,Dn
	{ "00000110 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ADDI),addi_imm_ea_byte,OP_CYCLES(12,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },				// ADDI.B #<data>,<ea>
	{ "00000110 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ADDI),addi_imm_ea_word,OP_CYCLES(12,2,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },				// ADDI.W #<data>,<ea>
	{ "00000110 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ADDI),addi_imm_ea_long,OP_CYCLES(20,3,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },				// ADDI.L #<data>,<ea>
	{ "0101ddd0 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ADDQ),addq_imm_dn_byte,OP_CYCLES(4,1,0),TBL_DN_1|TBL_DATA_2 },					// ADDQ.B #<data>,Dn
	{ "0101ddd0 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ADDQ),addq_imm_dn_word,OP_CYCLES(4,1,0),TBL_DN_1|TBL_DATA_2 },					// ADDQ.W #<data>,Dn
	{ "0101ddd0 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ADDQ),addq_imm_dn_long,OP_CYCLES(8,1,0),TBL_DN_1|TBL_DATA_2 },					// ADDQ.L #<data>,Dn
	{ "0101ddd0 01eeeeee", EFFADDR_0100000_000_00,	DISFUNC(Disass_ADDQ),addq_imm_an_word,OP_CYCLES(8,1,0),TBL_AN_1|TBL_DATA_2 },					// ADDQ.W #<data>,An	(no byte access for An)
	{ "0101ddd0 10eeeeee", EFFADDR_0100000_000_00,	DISFUNC(Disass_ADDQ),addq_imm_an_long,OP_CYCLES(8,1,0),TBL_AN_1|TBL_DATA_2 },					// ADDQ.L #<data>,An
	{ "0101ddd0 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ADDQ),addq_imm_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DATA_2 },	// ADDQ.B #<data>,<ea>
	{ "0101ddd0 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ADDQ),addq_imm_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DATA_2 },	// ADDQ.W #<data>,<ea>
	{ "0101ddd0 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ADDQ),addq_imm_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DATA_2 },	// ADDQ.L #<data>,<ea>
	{ "1101xxx1 00000yyy", NULL,					DISFUNC(Disass_ADDX_Dy_Dx),addx_dn_dn_byte,OP_CYCLES(4,1,0),TBL_DNUPR_1|TBL_DN_2 },			// ADDX.B Dy,Dx
	{ "1101xxx1 01000yyy", NULL,					DISFUNC(Disass_ADDX_Dy_Dx),addx_dn_dn_word,OP_CYCLES(4,1,0),TBL_DNUPR_1|TBL_DN_2 },			// ADDX.W Dy,Dx
	{ "1101xxx1 10000yyy", NULL,					DISFUNC(Disass_ADDX_Dy_Dx),addx_dn_dn_long,OP_CYCLES(8,1,0),TBL_DNUPR_1|TBL_DN_2 },			// ADDX.L Dy,Dx
	{ "1101xxx1 00001yyy", NULL,					DISFUNC(Disass_ADDX_Ay_Ax),addx_an_an_byte,OP_CYCLES(18,3,1),TBL_ANUPR_1|TBL_AN_2 },			// ADDX.B -(Ay),-(Ax)
	{ "1101xxx1 01001yyy", NULL,					DISFUNC(Disass_ADDX_Ay_Ax),addx_an_an_word,OP_CYCLES(18,3,1),TBL_ANUPR_1|TBL_AN_2 },			// ADDX.W -(Ay),-(Ax)
	{ "1101xxx1 10001yyy", NULL,					DISFUNC(Disass_ADDX_Ay_Ax),addx_an_an_long,OP_CYCLES(30,5,2),TBL_ANUPR_1|TBL_AN_2 },			// ADDX.L -(Ay),-(Ax)
	{ "1100rrr0 00eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_AND_ea_Dn),and_ea_dn_byte,OP_CYCLES(4,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// AND.B <ea>,Dn
	{ "1100rrr0 01eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_AND_ea_Dn),and_ea_dn_word,OP_CYCLES(4,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// AND.W <ea>,Dn
	{ "1100rrr0 10eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_AND_ea_Dn),and_ea_dn_long,OP_CYCLES(6,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// AND.L <ea>,Dn
	{ "1100rrr1 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_AND_Dn_ea),and_dn_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// AND.B Dn,<ea>
	{ "1100rrr1 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_AND_Dn_ea),and_dn_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// AND.W Dn,<ea>
	{ "1100rrr1 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_AND_Dn_ea),and_dn_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// AND.L Dn,<ea>
	{ "00000010 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ANDI),andi_imm_dn_byte,OP_CYCLES(8,2,0),TBL_DN_1 },						// ANDI.B #<data>,Dn
	{ "00000010 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ANDI),andi_imm_dn_word,OP_CYCLES(8,2,0),TBL_DN_1 },						// ANDI.W #<data>,Dn
	{ "00000010 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ANDI),andi_imm_dn_long,OP_CYCLES(16,3,0),TBL_DN_1 },						// ANDI.L #<data>,Dn
	{ "00000010 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ANDI),andi_imm_ea_byte,OP_CYCLES(12,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },			// ANDI.B #<data>,<ea>
	{ "00000010 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ANDI),andi_imm_ea_word,OP_CYCLES(12,2,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },			// ANDI.W #<data>,<ea>
	{ "00000010 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ANDI),andi_imm_ea_long,OP_CYCLES(20,3,1),TBL_EA_LONG|TBL_EA_ADDCYCLES },			// ANDI.L #<data>,<ea>
	{ "00000010 00111100", NULL,					DISFUNC(Disass_ANDI_to_CCR),andi_imm_ccr_byte,OP_CYCLES(20,3,0),NULL },				// ANDI to CCR #<data>,CCR
	{ "00000010 01111100", NULL,					DISFUNC(Disass_ANDI_to_SR),andi_imm_sr_word,OP_CYCLES(20,3,0),NULL },				// ANDI to SR #<data>,SR
	{ "1110xxx1 00100yyy", NULL,					DISFUNC(Disass_ASL_Dx_Dy),asl_dn_dn_byte,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// ASL.B Dx,Dy
	{ "1110xxx1 01100yyy", NULL,					DISFUNC(Disass_ASL_Dx_Dy),asl_dn_dn_word,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// ASL.W Dx,Dy
	{ "1110xxx1 10100yyy", NULL,					DISFUNC(Disass_ASL_Dx_Dy),asl_dn_dn_long,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// ASL.L Dx,Dy
	{ "1110ddd1 00000yyy", NULL,					DISFUNC(Disass_ASL_data_Dy),asl_imm_dn_byte,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ASL.B #<data>,Dy
	{ "1110ddd1 01000yyy", NULL,					DISFUNC(Disass_ASL_data_Dy),asl_imm_dn_word,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ASL.W #<data>,Dy
	{ "1110ddd1 10000yyy", NULL,					DISFUNC(Disass_ASL_data_Dy),asl_imm_dn_long,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ASL.L #<data>,Dy
	{ "11100001 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ASL),asl_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },				// ASL.W <ea>
	{ "1110xxx0 00100yyy", NULL,					DISFUNC(Disass_ASR_Dx_Dy),asr_dn_dn_byte,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// ASR.B Dx,Dy
	{ "1110xxx0 01100yyy", NULL,					DISFUNC(Disass_ASR_Dx_Dy),asr_dn_dn_word,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// ASR.W Dx,Dy
	{ "1110xxx0 10100yyy", NULL,					DISFUNC(Disass_ASR_Dx_Dy),asr_dn_dn_long,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// ASR.L Dx,Dy
	{ "1110ddd0 00000yyy", NULL,					DISFUNC(Disass_ASR_data_Dy),asr_imm_dn_byte,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ASR.B #<data>,Dy
	{ "1110ddd0 01000yyy", NULL,					DISFUNC(Disass_ASR_data_Dy),asr_imm_dn_word,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ASR.W #<data>,Dy
	{ "1110ddd0 10000yyy", NULL,					DISFUNC(Disass_ASR_data_Dy),asr_imm_dn_long,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ASR.L #<data>,Dy
	{ "11100000 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ASR),asr_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },				// ASR.W <ea>
	{ "01100100 oooooooo", NULL,					DISFUNC(Disass_Bcc),bcc_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BCC.S <label>
	{ "01100100 00000000", NULL,					DISFUNC(Disass_Bcc),bcc_w,OP_CYCLES(12,1,0),NULL },									// BCC.W <label>
	{ "01100101 oooooooo", NULL,					DISFUNC(Disass_Bcc),bcs_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BCS.S <label>
	{ "01100101 00000000", NULL,					DISFUNC(Disass_Bcc),bcs_w,OP_CYCLES(12,1,0),NULL },									// BCS.W <label>
	{ "01100111 oooooooo", NULL,					DISFUNC(Disass_Bcc),beq_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BEQ.S <label>
	{ "01100111 00000000", NULL,					DISFUNC(Disass_Bcc),beq_w,OP_CYCLES(12,1,0),NULL },									// BEQ.W <label>
	{ "01101100 oooooooo", NULL,					DISFUNC(Disass_Bcc),bge_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BGE.S <label>
	{ "01101100 00000000", NULL,					DISFUNC(Disass_Bcc),bge_w,OP_CYCLES(12,1,0),NULL },									// BGE.W <label>
	{ "01101110 oooooooo", NULL,					DISFUNC(Disass_Bcc),bgt_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BGT.S <label>
	{ "01101110 00000000", NULL,					DISFUNC(Disass_Bcc),bgt_w,OP_CYCLES(12,1,0),NULL },									// BGT.W <label>
	{ "01100010 oooooooo", NULL,					DISFUNC(Disass_Bcc),bhi_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BHI.S <label>
	{ "01100010 00000000", NULL,					DISFUNC(Disass_Bcc),bhi_w,OP_CYCLES(12,1,0),NULL },									// BHI.W <label>
	{ "01101111 oooooooo", NULL,					DISFUNC(Disass_Bcc),ble_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BLE.S <label>
	{ "01101111 00000000", NULL,					DISFUNC(Disass_Bcc),ble_w,OP_CYCLES(12,1,0),NULL },									// BLE.W <label>
	{ "01100011 oooooooo", NULL,					DISFUNC(Disass_Bcc),bls_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BLS.S <label>
	{ "01100011 00000000", NULL,					DISFUNC(Disass_Bcc),bls_w,OP_CYCLES(12,1,0),NULL },									// BLS.W <label>
	{ "01101101 oooooooo", NULL,					DISFUNC(Disass_Bcc),blt_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BLT.S <label>
	{ "01101101 00000000", NULL,					DISFUNC(Disass_Bcc),blt_w,OP_CYCLES(12,1,0),NULL },									// BLT.W <label>
	{ "01101011 oooooooo", NULL,					DISFUNC(Disass_Bcc),bmi_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BMI.S <label>
	{ "01101011 00000000", NULL,					DISFUNC(Disass_Bcc),bmi_w,OP_CYCLES(12,1,0),NULL },									// BMI.W <label>
	{ "01100110 oooooooo", NULL,					DISFUNC(Disass_Bcc),bne_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BNE.S <label>
	{ "01100110 00000000", NULL,					DISFUNC(Disass_Bcc),bne_w,OP_CYCLES(12,1,0),NULL },									// BNE.W <label>
	{ "01101010 oooooooo", NULL,					DISFUNC(Disass_Bcc),bpl_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BPL.S <label>
	{ "01101010 00000000", NULL,					DISFUNC(Disass_Bcc),bpl_w,OP_CYCLES(12,1,0),NULL },									// BPL.W <label>
	{ "01101000 oooooooo", NULL,					DISFUNC(Disass_Bcc),bvc_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BVC.S <label>
	{ "01101000 00000000", NULL,					DISFUNC(Disass_Bcc),bvc_w,OP_CYCLES(12,1,0),NULL },									// BVC.W <label>
	{ "01101001 oooooooo", NULL,					DISFUNC(Disass_Bcc),bvs_s,OP_CYCLES(8,1,0),TBL_SOFFSET_1 },									// BVS.S <label>
	{ "01101001 00000000", NULL,					DISFUNC(Disass_Bcc),bvs_w,OP_CYCLES(12,1,0),NULL },									// BVS.W <label>
	{ "0000rrr1 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_BCHG),bchg_dn_dn_long,OP_CYCLES(10,1,0),TBL_DN_1|TBL_DNUPR_2 },						// BCHG.L Dn,Dn
	{ "0000rrr1 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_BCHG),bchg_dn_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// BCHG.B Dn,<ea>
	{ "00001000 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_BCHG_imm),bchg_imm_dn_long,OP_CYCLES(14,2,0),TBL_DN_1 },					// BCHG.L #<data>,Dn
	{ "00001000 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_BCHG_imm),bchg_imm_ea_byte,OP_CYCLES(12,2,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES },		// BCHG.B #<data>,<ea>
	{ "0000rrr1 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_BCLR),bclr_dn_dn_long,OP_CYCLES(10,1,0),TBL_DN_1|TBL_DNUPR_2 },						// BCLR.L Dn,Dn
	{ "0000rrr1 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_BCLR),bclr_dn_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// BCLR.B Dn,<ea>
	{ "00001000 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_BCLR_imm),bclr_imm_dn_long,OP_CYCLES(14,2,0),TBL_DN_1 },					// BCLR.L #<data>,Dn
	{ "00001000 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_BCLR_imm),bclr_imm_ea_byte,OP_CYCLES(12,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },		// BCLR.B #<data>,<ea>	
	{ "01100000 oooooooo", NULL,					DISFUNC(Disass_BRA),bra_s,OP_CYCLES(10,2,0),TBL_SOFFSET_1 },									// BRA.S <label>
	{ "01100000 00000000", NULL,					DISFUNC(Disass_BRA),bra_w,OP_CYCLES(10,2,0),NULL },									// BRA.W <label>
	{ "0000rrr1 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_BSET),bset_dn_dn_long,OP_CYCLES(8,1,0),TBL_DN_1|TBL_DNUPR_2 },						// BSET.L Dn,Dn
	{ "0000rrr1 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_BSET),bset_dn_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// BSET.B Dn,<ea>
	{ "00001000 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_BSET_imm),bset_imm_dn_long,OP_CYCLES(14,2,0),TBL_DN_1 },					// BSET.L #<data>,Dn
	{ "00001000 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_BSET_imm),bset_imm_ea_byte,OP_CYCLES(12,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },		// BSET.B #<data>,<ea>
	{ "01100001 oooooooo", NULL,					DISFUNC(Disass_BSR),bsr_s,OP_CYCLES(18,2,0),TBL_SOFFSET_1 },						// BSR.S <label>
	{ "01100001 00000000", NULL,					DISFUNC(Disass_BSR),bsr_w,OP_CYCLES(18,2,2),NULL },						// BSR.W <label>
	{ "0000rrr1 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_BTST),btst_dn_dn_long,OP_CYCLES(6,1,0),TBL_DN_1|TBL_DNUPR_2 },						// BTST.L Dn,Dn
	{ "0000rrr1 00eeeeee", EFFADDR_0011111_111_11,	DISFUNC(Disass_BTST),btst_dn_ea_byte,OP_CYCLES(4,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// BTST.B Dn,<ea>
	{ "00001000 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_BTST_imm),btst_imm_dn_long,OP_CYCLES(10,2,0),TBL_DN_1 },					// BTST.L #<data>,Dn
	{ "00001000 00eeeeee", EFFADDR_0011111_110_11,	DISFUNC(Disass_BTST_imm),btst_imm_ea_byte,OP_CYCLES(8,2,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES },		// BTST.B #<data>,<ea>
	{ "0100rrr1 10eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_CHK),chk_ea_dn_word,OP_CYCLES(10,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },							// CHK.W <ea>,Dn
	{ "01000010 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_CLR),clr_dn_byte,OP_CYCLES(4,1,0),TBL_DN_1 },								// CLR.B Dn
	{ "01000010 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_CLR),clr_dn_word,OP_CYCLES(4,1,0),TBL_DN_1 },								// CLR.W Dn
	{ "01000010 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_CLR),clr_dn_long,OP_CYCLES(6,1,0),TBL_DN_1 },								// CLR.L Dn
	{ "01000010 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_CLR),clr_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },				// CLR.B <ea>
	{ "01000010 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_CLR),clr_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },				// CLR.W <ea>
	{ "01000010 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_CLR),clr_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },				// CLR.L <ea>
	{ "1011rrr0 00eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_CMP),cmp_ea_dn_byte,OP_CYCLES(4,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },				// CMP.B <ea>,Dn
	{ "1011rrr0 01eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_CMP),cmp_ea_dn_word,OP_CYCLES(4,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },				// CMP.W <ea>,Dn	(no byte access for An)
	{ "1011rrr0 10eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_CMP),cmp_ea_dn_long,OP_CYCLES(6,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },				// CMP.L <ea>,Dn
	{ "1011rrr0 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_CMPA),cmpa_ea_an_word,OP_CYCLES(6,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// CMPA.W <ea>,An
	{ "1011rrr1 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_CMPA),cmpa_ea_an_long,OP_CYCLES(6,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// CMPA.L <ea>,An
	{ "00001100 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_CMPI),cmpi_imm_dn_byte,OP_CYCLES(8,2,0),TBL_DN_1 },						// CMPI.B #<data>,Dn
	{ "00001100 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_CMPI),cmpi_imm_dn_word,OP_CYCLES(8,2,0),TBL_DN_1 },						// CMPI.W #<data>,Dn
	{ "00001100 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_CMPI),cmpi_imm_dn_long,OP_CYCLES(14,3,0),TBL_DN_1 },						// CMPI.L #<data>,Dn
	{ "00001100 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_CMPI),cmpi_imm_ea_byte,OP_CYCLES(8,2,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES },			// CMPI.B #<data>,<ea>
	{ "00001100 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_CMPI),cmpi_imm_ea_word,OP_CYCLES(8,2,0),TBL_EA_WORD|TBL_EA_ADDCYCLES },			// CMPI.W #<data>,<ea>
	{ "00001100 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_CMPI),cmpi_imm_ea_long,OP_CYCLES(12,3,0),TBL_EA_LONG|TBL_EA_ADDCYCLES },			// CMPI.L #<data>,<ea>
	{ "1011xxx1 00001yyy", NULL,					DISFUNC(Disass_CMPM),cmp_an_an_byte,OP_CYCLES(12,3,0),TBL_ANUPR_1|TBL_AN_2 },						// CMPM.B (Ay)+,(Ax)+
	{ "1011xxx1 01001yyy", NULL,					DISFUNC(Disass_CMPM),cmp_an_an_word,OP_CYCLES(12,3,0),TBL_ANUPR_1|TBL_AN_2 },						// CMPM.W (Ay)+,(Ax)+
	{ "1011xxx1 10001yyy", NULL,					DISFUNC(Disass_CMPM),cmp_an_an_long,OP_CYCLES(20,5,0),TBL_ANUPR_1|TBL_AN_2 },						// CMPM.L (Ay)+,(Ax)+
	{ "01010100 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbcc,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBCC Dn,<label>
	{ "01010101 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbcs,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBCS Dn,<label>
	{ "01010111 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbeq,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBEQ Dn,<label>
	{ "01010001 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbf,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBF Dn,<label>
	{ "01011100 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbge,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBGE Dn,<label>
	{ "01011110 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbgt,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBGT Dn,<label>
	{ "01010010 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbhi,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBHI Dn,<label>
	{ "01011111 11001rrr", NULL,					DISFUNC(Disass_DBcc),dble,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBLE Dn,<label>
	{ "01010011 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbls,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBLS Dn,<label>
	{ "01011101 11001rrr", NULL,					DISFUNC(Disass_DBcc),dblt,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBLT Dn,<label>
	{ "01011011 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbmi,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBMI Dn,<label>
	{ "01010110 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbne,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBNE Dn,<label>
	{ "01011010 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbpl,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBPL Dn,<label>
	{ "01010000 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbt,OP_CYCLES(12,2,0),TBL_DN_1 },									// DBT Dn,<label>
	{ "01011000 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbvc,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBVC Dn,<label>
	{ "01011001 11001rrr", NULL,					DISFUNC(Disass_DBcc),dbvs,OP_CYCLES(10,2,0),TBL_DN_1 },									// DBVS Dn,<label>
	{ "1000rrr1 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_DIVS),divs_ea_dn_word,OP_CYCLES(158,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// DIVS <ea>,Dn
	{ "1000rrr0 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_DIVU),divu_ea_dn_word,OP_CYCLES(140,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// DIVU <ea>,Dn
	{ "1011rrr1 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_EOR),eor_dn_dn_byte,OP_CYCLES(4,1,0),TBL_DN_1|TBL_DNUPR_2 },							// EOR.B Dn,Dn
	{ "1011rrr1 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_EOR),eor_dn_dn_word,OP_CYCLES(4,1,0),TBL_DN_1|TBL_DNUPR_2 },							// EOR.W Dn,Dn
	{ "1011rrr1 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_EOR),eor_dn_dn_long,OP_CYCLES(8,1,0),TBL_DN_1|TBL_DNUPR_2 },							// EOR.L Dn,Dn
	{ "1011rrr1 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_EOR),eor_dn_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },				// EOR.B Dn,<ea>
	{ "1011rrr1 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_EOR),eor_dn_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },				// EOR.W Dn,<ea>
	{ "1011rrr1 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_EOR),eor_dn_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// EOR.L Dn,<ea>
	{ "00001010 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_EORI),eori_imm_dn_byte,OP_CYCLES(8,2,0),TBL_DN_1 },						// EORI.B #<data>,Dn
	{ "00001010 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_EORI),eori_imm_dn_word,OP_CYCLES(8,2,0),TBL_DN_1 },						// EORI.W #<data>,Dn
	{ "00001010 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_EORI),eori_imm_dn_long,OP_CYCLES(16,3,0),TBL_DN_1 },						// EORI.L #<data>,Dn
	{ "00001010 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_EORI),eori_imm_ea_byte,OP_CYCLES(12,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },			// EORI.B #<data>,<ea>
	{ "00001010 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_EORI),eori_imm_ea_word,OP_CYCLES(12,2,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },			// EORI.W #<data>,<ea>
	{ "00001010 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_EORI),eori_imm_ea_long,OP_CYCLES(20,3,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },			// EORI.L #<data>,<ea>
	{ "00001010 00111100", NULL,					DISFUNC(Disass_EORI_to_CCR),eori_imm_ccr_byte,OP_CYCLES(20,3,0),NULL },				// EOR to CCR #<data>,CCR
	{ "00001010 01111100", NULL,					DISFUNC(Disass_EORI_to_SR),eori_imm_sr_word,OP_CYCLES(20,3,0),NULL },				// EOR to SR #<data>,SR
	{ "1100xxx1 01000yyy", NULL,					DISFUNC(Disass_EXG_Dx_Dy),exg_dn_dn_long,OP_CYCLES(6,1,0),TBL_DNUPR_1|TBL_DN_2 },					// EXG Dx,Dy
	{ "1100xxx1 01001yyy", NULL,					DISFUNC(Disass_EXG_Ax_Ay),exg_an_an_long,OP_CYCLES(6,1,0),TBL_ANUPR_1|TBL_AN_2 },					// EXG Ax,Ay
	{ "1100xxx1 10001yyy", NULL,					DISFUNC(Disass_EXG_Dx_Ay),exg_dn_an_long,OP_CYCLES(6,1,0),TBL_DNUPR_1|TBL_AN_2 },					// EXG Dx,Ay
	{ "01001000 10000rrr", NULL,					DISFUNC(Disass_EXT),ext_dn_word,OP_CYCLES(4,1,0),TBL_DN_1 },								// EXT.W Dn
	{ "01001000 11000rrr", NULL,					DISFUNC(Disass_EXT),ext_dn_long,OP_CYCLES(4,1,0),TBL_DN_1 },								// EXT.L Dn
	{ "01001010 11111100", NULL,					DISFUNC(Disass_ILLEGAL),illegal,OP_CYCLES(34,4,3),NULL },							// ILLEGAL
	{ "01001110 11eeeeee", EFFADDR_0010011_110_11,	DISFUNC(Disass_JMP),jmp_ea,OP_CYCLES(4,1,0),TBL_EA_LOAD|TBL_EA_ADDCYCLES },						// JMP <ea>
	{ "01001110 10eeeeee", EFFADDR_0010011_110_11,	DISFUNC(Disass_JSR),jsr_ea,OP_CYCLES(10,1,2),TBL_EA_LOAD|TBL_EA_ADDCYCLES },			// JSR <ea>
	{ "01001110 10111001", NULL,					DISFUNC(Disass_JSR),jsr_ea,OP_CYCLES(20,1,2),TBL_EA_LOAD },						// JSR <ea>(.L), correct cycle count
	{ "0100rrr1 11eeeeee", EFFADDR_0010011_110_11,	DISFUNC(Disass_LEA),lea_ea_an_long,OP_CYCLES(0,0,0),TBL_EA_LOAD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },				// LEA <ea>,An
	{ "01001110 01010rrr", NULL,					DISFUNC(Disass_LINK),link,OP_CYCLES(16,2,2),TBL_AN_1 },									// LINK An,#<displacement>
	{ "1110xxx1 00101yyy", NULL,					DISFUNC(Disass_LSL_Dx_Dy),lsl_dn_dn_byte,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// LSL.B Dx,Dy
	{ "1110xxx1 01101yyy", NULL,					DISFUNC(Disass_LSL_Dx_Dy),lsl_dn_dn_word,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// LSL.W Dx,Dy
	{ "1110xxx1 10101yyy", NULL,					DISFUNC(Disass_LSL_Dx_Dy),lsl_dn_dn_long,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// LSL.L Dx,Dy
	{ "1110ddd1 00001yyy", NULL,					DISFUNC(Disass_LSL_data_Dy),lsl_imm_dn_byte,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// LSL.B #<data>,Dy
	{ "1110ddd1 01001yyy", NULL,					DISFUNC(Disass_LSL_data_Dy),lsl_imm_dn_word,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// LSL.W #<data>,Dy
	{ "1110ddd1 10001yyy", NULL,					DISFUNC(Disass_LSL_data_Dy),lsl_imm_dn_long,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// LSL.L #<data>,Dy
	{ "11100011 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_LSL),lsl_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },				// LSL.W <ea>
	{ "1110xxx0 00101yyy", NULL,					DISFUNC(Disass_LSR_Dx_Dy),lsr_dn_dn_byte,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// LSR.B Dx,Dy
	{ "1110xxx0 01101yyy", NULL,					DISFUNC(Disass_LSR_Dx_Dy),lsr_dn_dn_word,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// LSR.W Dx,Dy
	{ "1110xxx0 10101yyy", NULL,					DISFUNC(Disass_LSR_Dx_Dy),lsr_dn_dn_long,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },					// LSR.L Dx,Dy
	{ "1110xxx0 00001yyy", NULL,					DISFUNC(Disass_LSR_data_Dy),lsr_imm_dn_byte,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// LSR.B #<data>,Dy
	{ "1110xxx0 01001yyy", NULL,					DISFUNC(Disass_LSR_data_Dy),lsr_imm_dn_word,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// LSR.W #<data>,Dy
	{ "1110xxx0 10001yyy", NULL,					DISFUNC(Disass_LSR_data_Dy),lsr_imm_dn_long,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// LSR.L #<data>,Dy
	{ "11100010 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_LSR),lsr_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },				// LSR.W <ea>
	{ "0001rrr0 00eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_000_byte,OP_CYCLES(4,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// MOVE.B <ea>,Dn	(no byte access for An...)
	{ "0011rrr0 00eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_000_word,OP_CYCLES(4,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// MOVE.W <ea>,Dn
	{ "0010rrr0 00eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_000_long,OP_CYCLES(4,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },			// MOVE.L <ea>,Dn
	{ "0001rrr0 10eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_010_byte,OP_CYCLES(8,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.B <ea>,(An)
	{ "0011rrr0 10eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_010_word,OP_CYCLES(8,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.W <ea>,(An)
	{ "0010rrr0 10eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_010_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.L <ea>,(An)
	{ "0001rrr0 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_011_byte,OP_CYCLES(8,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.B <ea>,(An)+
	{ "00011110 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_011_byte_A7,OP_CYCLES(8,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },		// MOVE.B <ea>,(A7)+
	{ "0011rrr0 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_011_word,OP_CYCLES(8,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.W <ea>,(An)+
	{ "0010rrr0 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_011_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.L <ea>,(An)+
	{ "0001rrr1 00eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_100_byte,OP_CYCLES(8,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.B <ea>,-(An)
	{ "00011111 00eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_100_byte_A7,OP_CYCLES(8,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },		// MOVE.B <ea>,-(A7)
	{ "0011rrr1 00eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_100_word,OP_CYCLES(8,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.W <ea>,-(An)
	{ "0010rrr1 00eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_100_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.L <ea>,-(An)
	{ "0001rrr1 01eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_101_byte,OP_CYCLES(12,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.B <ea>,(d16,An)
	{ "0011rrr1 01eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_101_word,OP_CYCLES(12,2,1),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.W <ea>,(d16,An)
	{ "0010rrr1 01eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_101_long,OP_CYCLES(16,2,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.L <ea>,(d16,An)
	{ "0001rrr1 10eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_110_byte,OP_CYCLES(14,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.B <ea>,(d8,An,Xn)
	{ "0011rrr1 10eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_110_word,OP_CYCLES(14,2,1),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.W <ea>,(d8,An,Xn)
	{ "0010rrr1 10eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_110_long,OP_CYCLES(18,2,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// MOVE.L <ea>,(d8,An,Xn)
	{ "00010001 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_111_000_byte,OP_CYCLES(12,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },		// MOVE.B <ea>,(xxx).W
	{ "00110001 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_111_000_word,OP_CYCLES(12,2,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },		// MOVE.W <ea>,(xxx).W
	{ "00100001 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_111_000_long,OP_CYCLES(16,2,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },		// MOVE.L <ea>,(xxx).W
	{ "00010011 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE),move_ea_111_001_byte,OP_CYCLES(16,3,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },		// MOVE.B <ea>,(xxx).L
	{ "00110011 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_111_001_word,OP_CYCLES(16,3,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },		// MOVE.W <ea>,(xxx).L
	{ "00100011 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVE),move_ea_111_001_long,OP_CYCLES(20,3,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },		// MOVE.L <ea>,(xxx).L
	{ "01000100 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE_to_CCR),move_ea_ccr_word,OP_CYCLES(12,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES },			// MOVE <ea>,CCR
	{ "01000000 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_MOVE_from_SR),move_sr_dn_word,OP_CYCLES(6,1,0),TBL_DN_1 },								// MOVE SR,Dn
	{ "01000000 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_MOVE_from_SR),move_sr_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },			// MOVE SR,<ea>
	{ "01000110 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MOVE_to_SR),move_ea_sr_word,OP_CYCLES(12,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES },			// MOVE <ea>,SR
	{ "01001110 01101rrr", NULL,					DISFUNC(Disass_MOVE_USP_An),move_usp_an_long,OP_CYCLES(4,1,0),TBL_AN_1 },								// MOVE USP,An
	{ "01001110 01100rrr", NULL,					DISFUNC(Disass_MOVE_An_USP),move_an_usp_long,OP_CYCLES(4,1,0),TBL_AN_1 },								// MOVE An,USP
	{ "0011rrr0 01eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVEA),movea_ea_an_word,OP_CYCLES(4,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },		// MOVEA.W <ea>,An
	{ "0010rrr0 01eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_MOVEA),movea_ea_an_long,OP_CYCLES(4,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },		// MOVEA.L <ea>,An
	{ "01001000 10eeeeee", EFFADDR_0010000_000_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_word,OP_CYCLES(8,0,0),TBL_EA_LOAD },							// MOVEM.W regs,(An)
	{ "01001000 10eeeeee", EFFADDR_0000100_000_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_an_dec_word,OP_CYCLES(8,0,0),TBL_AN_1 },						// MOVEM.W regs,-(An)
	{ "01001000 10eeeeee", EFFADDR_0000010_000_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_word,OP_CYCLES(12,0,0),TBL_EA_LOAD },						// MOVEM.W regs,(d16,An)
	{ "01001000 10eeeeee", EFFADDR_0000001_000_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_word,OP_CYCLES(14,0,0),TBL_EA_LOAD },						// MOVEM.W regs,(d8,An,Xn)
	{ "01001000 10eeeeee", EFFADDR_0000000_100_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_word,OP_CYCLES(12,0,0),TBL_EA_LOAD },						// MOVEM.W regs,(xxx).W
	{ "01001000 10eeeeee", EFFADDR_0000000_010_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_word,OP_CYCLES(16,0,0),TBL_EA_LOAD },			// MOVEM.W regs,(xxx).L
	{ "01001000 11eeeeee", EFFADDR_0010000_000_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_long,OP_CYCLES(8,0,0),TBL_EA_LOAD },				// MOVEM.L regs,<ea>
	{ "01001000 11eeeeee", EFFADDR_0000100_000_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_an_dec_long,OP_CYCLES(8,0,0),TBL_AN_1 },			// MOVEM.L regs,-(An)
	{ "01001000 11eeeeee", EFFADDR_0000010_000_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_long,OP_CYCLES(8,0,0),TBL_EA_LOAD },				// MOVEM.L regs,<ea>
	{ "01001000 11eeeeee", EFFADDR_0000001_000_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_long,OP_CYCLES(8,0,0),TBL_EA_LOAD },				// MOVEM.L regs,<ea>
	{ "01001000 11eeeeee", EFFADDR_0000000_100_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_long,OP_CYCLES(8,0,0),TBL_EA_LOAD },				// MOVEM.L regs,<ea>
	{ "01001000 11eeeeee", EFFADDR_0000000_010_00,	DISFUNC(Disass_MOVEM_regs_ea),movem_reg_ea_long,OP_CYCLES(8,0,0),TBL_EA_LOAD },				// MOVEM.L regs,<ea>
	{ "01001100 10eeeeee", EFFADDR_0010000_000_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_word,OP_CYCLES(12,0,0),TBL_EA_LOAD },			// MOVEM.W <ea>,(An)
	{ "01001100 10eeeeee", EFFADDR_0001000_000_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_an_inc_reg_word,OP_CYCLES(12,0,0),TBL_AN_1 },			// MOVEM.W (An)+,regs
	{ "01001100 10eeeeee", EFFADDR_0000010_000_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_word,OP_CYCLES(16,0,0),TBL_EA_LOAD },			// MOVEM.W <ea>,(d16,An)
	{ "01001100 10eeeeee", EFFADDR_0000001_000_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_word,OP_CYCLES(18,0,0),TBL_EA_LOAD },			// MOVEM.W <ea>,(d8,An,Xn)
	{ "01001100 10eeeeee", EFFADDR_0000000_100_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_word,OP_CYCLES(16,0,0),TBL_EA_LOAD },			// MOVEM.W <ea>,(xxx).W
	{ "01001100 10eeeeee", EFFADDR_0000000_010_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_word,OP_CYCLES(20,0,0),TBL_EA_LOAD },			// MOVEM.W <ea>,(xxx).L
	{ "01001100 10eeeeee", EFFADDR_0000000_000_10,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_word,OP_CYCLES(16,0,0),TBL_EA_LOAD },			// MOVEM.W <ea>,(d16,PC)
	{ "01001100 10eeeeee", EFFADDR_0000000_000_01,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_word,OP_CYCLES(18,0,0),TBL_EA_LOAD },			// MOVEM.W <ea>,(d8,PC,Xn)
	{ "01001100 11eeeeee", EFFADDR_0010000_000_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_long,OP_CYCLES(12,0,0),TBL_EA_LOAD },			// MOVEM.L <ea>,(An)
	{ "01001100 11eeeeee", EFFADDR_0001000_000_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_an_inc_reg_long,OP_CYCLES(12,0,0),TBL_AN_1 },			// MOVEM.L (An)+,regs
	{ "01001100 11eeeeee", EFFADDR_0000010_000_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_long,OP_CYCLES(16,0,0),TBL_EA_LOAD },			// MOVEM.L <ea>,(d16,An)
	{ "01001100 11eeeeee", EFFADDR_0000001_000_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_long,OP_CYCLES(18,0,0),TBL_EA_LOAD },			// MOVEM.L <ea>,(d8,An,Xn)
	{ "01001100 11eeeeee", EFFADDR_0000000_100_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_long,OP_CYCLES(16,0,0),TBL_EA_LOAD },			// MOVEM.L <ea>,(xxx).W
	{ "01001100 11eeeeee", EFFADDR_0000000_010_00,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_long,OP_CYCLES(20,0,0),TBL_EA_LOAD },			// MOVEM.L <ea>,(xxx).L
	{ "01001100 11eeeeee", EFFADDR_0000000_000_10,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_long,OP_CYCLES(16,0,0),TBL_EA_LOAD },			// MOVEM.L <ea>,(d16,PC)
	{ "01001100 11eeeeee", EFFADDR_0000000_000_01,	DISFUNC(Disass_MOVEM_ea_regs),movem_ea_reg_long,OP_CYCLES(18,0,0),TBL_EA_LOAD },			// MOVEM.L <ea>,(d8,PC,Xn)
	{ "0000xxx1 10001yyy", NULL,					DISFUNC(Disass_MOVEP_Dn_An),movep_dn_an_word,OP_CYCLES(16,2,2),TBL_DNUPR_1|TBL_AN_2 },				// MOVEP.W Dx,(d,Ay)
	{ "0000xxx1 11001yyy", NULL,					DISFUNC(Disass_MOVEP_Dn_An),movep_dn_an_long,OP_CYCLES(24,2,4),TBL_DNUPR_1|TBL_AN_2 },				// MOVEP.L Dx,(d,Ay)
	{ "0000xxx1 00001yyy", NULL,					DISFUNC(Disass_MOVEP_An_Dn),movep_an_dn_word,OP_CYCLES(16,4,0),TBL_DNUPR_1|TBL_AN_2 },				// MOVEP.W (d,Ay),Dx
	{ "0000xxx1 01001yyy", NULL,					DISFUNC(Disass_MOVEP_An_Dn),movep_an_dn_long,OP_CYCLES(24,6,0),TBL_DNUPR_1|TBL_AN_2 },				// MOVEP.L (d,Ay),Dx
	{ "0111rrr0 oooooooo", NULL,					DISFUNC(Disass_MOVEQ),moveq_imm_dn_long,OP_CYCLES(4,1,0),TBL_SOFFSET_1|TBL_DNUPR_2 },					// MOVEQ #<data>,Dn
	{ "1100rrr1 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MULS),muls_ea_dn_word,OP_CYCLES(70,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// MULS <ea>,Dn
	{ "1100rrr0 11eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_MULU),mulu_ea_dn_word,OP_CYCLES(70,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// MULU <ea>,Dn
	{ "01001000 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NBCD),nbcd_dn_byte,OP_CYCLES(6,1,0),TBL_DN_1 },											// NBCD Dn
	{ "01001000 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NBCD),nbcd_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// NBCD <ea>
	{ "01000100 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NEG),neg_dn_byte,OP_CYCLES(4,1,0),TBL_DN_1 },											// NEG.B Dn
	{ "01000100 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NEG),neg_dn_word,OP_CYCLES(4,1,0),TBL_DN_1 },											// NEG.W Dn
	{ "01000100 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NEG),neg_dn_long,OP_CYCLES(6,1,0),TBL_DN_1 },											// NEG.L Dn
	{ "01000100 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NEG),neg_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// NEG.B <ea>
	{ "01000100 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NEG),neg_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },						// NEG.W <ea>
	{ "01000100 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NEG),neg_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },						// NEG.L <ea>
	{ "01000000 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NEGX),negx_dn_byte,OP_CYCLES(4,1,0),TBL_DN_1 },											// NEGX.B Dn
	{ "01000000 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NEGX),negx_dn_word,OP_CYCLES(4,1,0),TBL_DN_1 },											// NEGX.W Dn
	{ "01000000 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NEGX),negx_dn_long,OP_CYCLES(6,1,0),TBL_DN_1 },											// NEGX.L Dn
	{ "01000000 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NEGX),negx_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// NEGX.B <ea>
	{ "01000000 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NEGX),negx_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },						// NEGX.W <ea>
	{ "01000000 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NEGX),negx_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },						// NEGX.L <ea>
	{ "01001110 01110001", NULL,					DISFUNC(Disass_NOP),nop,OP_CYCLES(4,1,0),NULL },														// NOP
	{ "01000110 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NOT),not_dn_byte,OP_CYCLES(4,1,0),TBL_DN_1 },											// NOT.B Dn
	{ "01000110 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NOT),not_dn_word,OP_CYCLES(4,1,0),TBL_DN_1 },											// NOT.W Dn
	{ "01000110 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_NOT),not_dn_long,OP_CYCLES(6,1,0),TBL_DN_1 },											// NOT.L Dn
	{ "01000110 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NOT),not_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// NOT.B <ea>
	{ "01000110 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NOT),not_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },						// NOT.W <ea>
	{ "01000110 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_NOT),not_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },						// NOT.L <ea>
	{ "1000rrr0 00eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_OR_ea_Dn),or_ea_dn_byte,OP_CYCLES(4,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// OR.B <ea>,Dn
	{ "1000rrr0 01eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_OR_ea_Dn),or_ea_dn_word,OP_CYCLES(4,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// OR.W <ea>,Dn
	{ "1000rrr0 10eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_OR_ea_Dn),or_ea_dn_long,OP_CYCLES(6,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// OR.L <ea>,Dn
	{ "1000rrr1 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_OR_Dn_ea),or_dn_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// OR.B Dn,<ea>
	{ "1000rrr1 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_OR_Dn_ea),or_dn_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// OR.W Dn,<ea>
	{ "1000rrr1 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_OR_Dn_ea),or_dn_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },	// OR.L Dn,<ea>
	{ "00000000 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ORI),ori_imm_dn_byte,OP_CYCLES(8,2,0),TBL_DN_1 },										// ORI.B #<data>,Dn
	{ "00000000 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ORI),ori_imm_dn_word,OP_CYCLES(8,2,0),TBL_DN_1 },										// ORI.W #<data>,Dn
	{ "00000000 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_ORI),ori_imm_dn_long,OP_CYCLES(16,3,0),TBL_DN_1 },										// ORI.L #<data>,Dn
	{ "00000000 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ORI),ori_imm_ea_byte,OP_CYCLES(12,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },					// ORI.B #<data>,<ea>
	{ "00000000 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ORI),ori_imm_ea_word,OP_CYCLES(12,2,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },					// ORI.W #<data>,<ea>
	{ "00000000 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ORI),ori_imm_ea_long,OP_CYCLES(20,3,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },					// ORI.L #<data>,<ea>
	{ "00000000 00111100", NULL,					DISFUNC(Disass_ORI_to_CCR),ori_imm_ccr_byte,OP_CYCLES(20,3,0),NULL },									// OR to CCR #<data>,CCR
	{ "00000000 01111100", NULL,					DISFUNC(Disass_ORI_to_SR),ori_imm_sr_word,OP_CYCLES(20,3,0),NULL },										// OR to SR #<data>,SR
	{ "01001000 01eeeeee", EFFADDR_0010011_110_11,	DISFUNC(Disass_PEA),pea_ea_long,OP_CYCLES(12,1,2),TBL_EA_LOAD|TBL_EA_ADDCYCLES },						// PEA <ea>
	{ "01001110 01110000", NULL,					DISFUNC(Disass_RESET),reset,OP_CYCLES(40,6,0),NULL },													// RESET
	{ "1110xxx1 00111yyy", NULL,					DISFUNC(Disass_ROL_Dx_Dy),rol_dn_dn_byte,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROL.B Dx,Dy
	{ "1110xxx1 01111yyy", NULL,					DISFUNC(Disass_ROL_Dx_Dy),rol_dn_dn_word,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROL.W Dx,Dy
	{ "1110xxx1 10111yyy", NULL,					DISFUNC(Disass_ROL_Dx_Dy),rol_dn_dn_long,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROL.L Dx,Dy
	{ "1110ddd1 00011yyy", NULL,					DISFUNC(Disass_ROL_data_Dy),rol_imm_dn_byte,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },						// ROL.B #<data>,Dy
	{ "1110ddd1 01011yyy", NULL,					DISFUNC(Disass_ROL_data_Dy),rol_imm_dn_word,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },						// ROL.W #<data>,Dy
	{ "1110ddd1 10011yyy", NULL,					DISFUNC(Disass_ROL_data_Dy),rol_imm_dn_long,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },						// ROL.L #<data>,Dy
	{ "11100111 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ROL),rol_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },						// ROL.W <ea>
	{ "1110xxx0 00111yyy", NULL,					DISFUNC(Disass_ROR_Dx_Dy),ror_dn_dn_byte,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROR.B Dx,Dy
	{ "1110xxx0 01111yyy", NULL,					DISFUNC(Disass_ROR_Dx_Dy),ror_dn_dn_word,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROR.W Dx,Dy
	{ "1110xxx0 10111yyy", NULL,					DISFUNC(Disass_ROR_Dx_Dy),ror_dn_dn_long,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROR.L Dx,Dy
	{ "1110ddd0 00011yyy", NULL,					DISFUNC(Disass_ROR_data_Dy),ror_imm_dn_byte,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },						// ROR.B #<data>,Dy
	{ "1110ddd0 01011yyy", NULL,					DISFUNC(Disass_ROR_data_Dy),ror_imm_dn_word,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },						// ROR.W #<data>,Dy
	{ "1110ddd0 10011yyy", NULL,					DISFUNC(Disass_ROR_data_Dy),ror_imm_dn_long,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },						// ROR.L #<data>,Dy
	{ "11100110 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ROR),ror_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },						// ROR.W <ea>
	{ "1110xxx1 00110yyy", NULL,					DISFUNC(Disass_ROXL_Dx_Dy),roxl_dn_dn_byte,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROXL.B Dx,Dy
	{ "1110xxx1 01110yyy", NULL,					DISFUNC(Disass_ROXL_Dx_Dy),roxl_dn_dn_word,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROXL.W Dx,Dy
	{ "1110xxx1 10110yyy", NULL,					DISFUNC(Disass_ROXL_Dx_Dy),roxl_dn_dn_long,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROXL.L Dx,Dy
	{ "1110ddd1 00010yyy", NULL,					DISFUNC(Disass_ROXL_data_Dy),roxl_imm_dn_byte,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ROXL.B #<data>,Dy
	{ "1110ddd1 01010yyy", NULL,					DISFUNC(Disass_ROXL_data_Dy),roxl_imm_dn_word,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ROXL.W #<data>,Dy
	{ "1110ddd1 10010yyy", NULL,					DISFUNC(Disass_ROXL_data_Dy),roxl_imm_dn_long,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ROXL.L #<data>,Dy
	{ "11100101 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ROXL),roxl_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },						// ROXL.W <ea>
	{ "1110xxx0 00110yyy", NULL,					DISFUNC(Disass_ROXR_Dx_Dy),roxr_dn_dn_byte,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROXR.B Dx,Dy
	{ "1110xxx0 01110yyy", NULL,					DISFUNC(Disass_ROXR_Dx_Dy),roxr_dn_dn_word,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROXR.W Dx,Dy
	{ "1110xxx0 10110yyy", NULL,					DISFUNC(Disass_ROXR_Dx_Dy),roxr_dn_dn_long,OP_CYCLES(0,0,0),TBL_DNUPR_1|TBL_DN_2 },						// ROXR.L Dx,Dy
	{ "1110ddd0 00010yyy", NULL,					DISFUNC(Disass_ROXR_data_Dy),roxr_imm_dn_byte,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ROXR.B #<data>,Dy
	{ "1110ddd0 01010yyy", NULL,					DISFUNC(Disass_ROXR_data_Dy),roxr_imm_dn_word,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ROXR.W #<data>,Dy
	{ "1110ddd0 10010yyy", NULL,					DISFUNC(Disass_ROXR_data_Dy),roxr_imm_dn_long,OP_CYCLES(0,0,0),TBL_DN_1|TBL_DATA_2 },					// ROXR.L #<data>,Dy
	{ "11100100 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_ROXR),roxr_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },						// ROXR.W <ea>
	{ "01001110 01110011", NULL,					DISFUNC(Disass_RTE),rte,OP_CYCLES(20,5,0),NULL },														// RTE
	{ "01001110 01110111", NULL,					DISFUNC(Disass_RTR),rtr,OP_CYCLES(20,5,0),NULL },														// RTR
	{ "01001110 01110101", NULL,					DISFUNC(Disass_RTS),rts,OP_CYCLES(16,4,0),NULL },														// RTS
	{ "1000xxx1 00000yyy", NULL,					DISFUNC(Disass_SBCD_Dy_Dx),sbcd_dn_dn,OP_CYCLES(6,1,0),TBL_DNUPR_1|TBL_DN_2 },							// SBCD Dy,Dx
	{ "1000xxx1 00001yyy", NULL,					DISFUNC(Disass_SBCD_Ay_Ax),sbcd_an_an,OP_CYCLES(18,3,1),TBL_ANUPR_1|TBL_AN_2 },							// SBCD -(Ay),-(Ax)
	{ "01010100 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),scc_dn,OP_CYCLES(4,1,0),TBL_DN_1 },													// SCC Dn
	{ "01010100 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),scc,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },								// SCC <ea>	
	{ "01010101 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),scs_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SCS Dn
	{ "01010101 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),scs,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SCS <ea>	
	{ "01010111 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),seq_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SEQ Dn
	{ "01010111 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),seq,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SEQ <ea>	
	{ "01010001 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),sf_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SF Dn
	{ "01010001 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),sf,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },							// SF <ea>	
	{ "01011100 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),sge_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SGE Dn
	{ "01011100 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),sge,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SGE <ea>	
	{ "01011110 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),sgt_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SGT Dn
	{ "01011110 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),sgt,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SGT <ea>	
	{ "01010010 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),shi_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SHI Dn
	{ "01010010 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),shi,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SHI <ea>	
	{ "01011111 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),sle_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SLE Dn
	{ "01011111 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),sle,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SLE <ea>	
	{ "01010011 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),sls_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SLS Dn
	{ "01010011 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),sls,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SLS <ea>	
	{ "01011101 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),slt_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SLT Dn
	{ "01011101 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),slt,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SLT <ea>	
	{ "01011011 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),smi_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SMI Dn
	{ "01011011 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),smi,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SMI <ea>	
	{ "01010110 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),sne_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SNE Dn
	{ "01010110 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),sne,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SNE <ea>	
	{ "01011010 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),spl_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SPL Dn
	{ "01011010 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),spl,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SPL <ea>	
	{ "01010000 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),st_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// ST Dn
	{ "01010000 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),st,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },							// ST <ea>	
	{ "01011000 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),svc_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SVC Dn
	{ "01011000 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),svc,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SVC <ea>	
	{ "01011001 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_Scc),svs_dn,OP_CYCLES(4,1,0),TBL_DN_1 },						// SVS Dn
	{ "01011001 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_Scc),svs,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },						// SVS <ea>	
	{ "01001110 01110010", NULL,					DISFUNC(Disass_STOP),stop,OP_CYCLES(4,0,0),NULL },									// STOP #<data>
	{ "1001rrr0 00eeeeee", EFFADDR_1011111_111_11,	DISFUNC(Disass_SUB_ea_Dn),sub_ea_dn_byte,OP_CYCLES(4,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// SUB.B <ea>,Dn	(no byte access)
	{ "1001rrr0 01eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_SUB_ea_Dn),sub_ea_dn_word,OP_CYCLES(4,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// SUB.W <ea>,Dn
	{ "1001rrr0 10eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_SUB_ea_Dn),sub_ea_dn_long,OP_CYCLES(6,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// SUB.L <ea>,Dn
	{ "1001rrr1 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_SUB_Dn_ea),sub_dn_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// SUB.B Dn,<ea>
	{ "1001rrr1 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_SUB_Dn_ea),sub_dn_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// SUB.W Dn,<ea>
	{ "1001rrr1 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_SUB_Dn_ea),sub_dn_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DNUPR_2 },		// SUB.L Dn,<ea>
	{ "1001rrr0 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_SUBA),suba_ea_an_word,OP_CYCLES(8,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// SUBA.W <ea>,An
	{ "1001rrr1 11eeeeee", EFFADDR_1111111_111_11,	DISFUNC(Disass_SUBA),suba_ea_an_long,OP_CYCLES(6,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_ANUPR_2 },			// SUBA.L <ea>,An
	{ "00000100 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_SUBI),subi_imm_dn_byte,OP_CYCLES(8,2,0),TBL_DN_1 },						// SUBI.B #<data>,Dn
	{ "00000100 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_SUBI),subi_imm_dn_word,OP_CYCLES(8,2,0),TBL_DN_1 },						// SUBI.W #<data>,Dn
	{ "00000100 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_SUBI),subi_imm_dn_long,OP_CYCLES(16,3,0),TBL_DN_1 },						// SUBI.L #<data>,Dn
	{ "00000100 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_SUBI),subi_imm_ea_byte,OP_CYCLES(12,2,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },			// SUBI.B #<data>,<ea>
	{ "00000100 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_SUBI),subi_imm_ea_word,OP_CYCLES(12,2,1),TBL_EA_WORD|TBL_EA_ADDCYCLES },			// SUBI.W #<data>,<ea>
	{ "00000100 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_SUBI),subi_imm_ea_long,OP_CYCLES(20,3,2),TBL_EA_LONG|TBL_EA_ADDCYCLES },			// SUBI.L #<data>,<ea>
	{ "0101ddd1 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_SUBQ),subq_imm_dn_byte,OP_CYCLES(4,1,0),TBL_DN_1|TBL_DATA_2 },						// SUBQ.B #<data>,Dn
	{ "0101ddd1 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_SUBQ),subq_imm_dn_word,OP_CYCLES(4,1,0),TBL_DN_1|TBL_DATA_2 },						// SUBQ.W #<data>,Dn
	{ "0101ddd1 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_SUBQ),subq_imm_dn_long,OP_CYCLES(8,1,0),TBL_DN_1|TBL_DATA_2 },						// SUBQ.L #<data>,Dn
	{ "0101ddd1 01eeeeee", EFFADDR_0100000_000_00,	DISFUNC(Disass_SUBQ),subq_imm_an_word,OP_CYCLES(8,1,0),TBL_AN_1|TBL_DATA_2 },						// SUBQ.W #<data>,An	(no byte access)
	{ "0101ddd1 10eeeeee", EFFADDR_0100000_000_00,	DISFUNC(Disass_SUBQ),subq_imm_an_long,OP_CYCLES(8,1,0),TBL_AN_1|TBL_DATA_2 },						// SUBQ.L #<data>,An
	{ "0101ddd1 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_SUBQ),subq_imm_ea_byte,OP_CYCLES(8,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES|TBL_DATA_2 },			// SUBQ.B #<data>,<ea>
	{ "0101ddd1 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_SUBQ),subq_imm_ea_word,OP_CYCLES(8,1,1),TBL_EA_WORD|TBL_EA_ADDCYCLES|TBL_DATA_2 },			// SUBQ.W #<data>,<ea>
	{ "0101ddd1 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_SUBQ),subq_imm_ea_long,OP_CYCLES(12,1,2),TBL_EA_LONG|TBL_EA_ADDCYCLES|TBL_DATA_2 },			// SUBQ.L #<data>,<ea>
	{ "1001xxx1 00000yyy", NULL,					DISFUNC(Disass_SUBX_Dy_Dx),subx_dn_dn_byte,OP_CYCLES(4,1,0),TBL_DNUPR_1|TBL_DN_2 },					// SUBX.B Dy,Dx
	{ "1001xxx1 01000yyy", NULL,					DISFUNC(Disass_SUBX_Dy_Dx),subx_dn_dn_word,OP_CYCLES(4,1,0),TBL_DNUPR_1|TBL_DN_2 },					// SUBX.W Dy,Dx
	{ "1001xxx1 10000yyy", NULL,					DISFUNC(Disass_SUBX_Dy_Dx),subx_dn_dn_long,OP_CYCLES(8,1,0),TBL_DNUPR_1|TBL_DN_2 },					// SUBX.L Dy,Dx
	{ "1001xxx1 00001yyy", NULL,					DISFUNC(Disass_SUBX_Ay_Ax),subx_an_an_byte,OP_CYCLES(18,3,1),TBL_ANUPR_1|TBL_AN_2 },					// SUBX.B -(Ay),-(Ax)
	{ "1001xxx1 01001yyy", NULL,					DISFUNC(Disass_SUBX_Ay_Ax),subx_an_an_word,OP_CYCLES(18,3,1),TBL_ANUPR_1|TBL_AN_2 },					// SUBX.W -(Ay),-(Ax)
	{ "1001xxx1 10001yyy", NULL,					DISFUNC(Disass_SUBX_Ay_Ax),subx_an_an_long,OP_CYCLES(30,5,2),TBL_ANUPR_1|TBL_AN_2 },					// SUBX.L -(Ay),-(Ax)
	{ "01001000 01000rrr", NULL,					DISFUNC(Disass_SWAP),swap_dn_word,OP_CYCLES(4,1,0),TBL_DN_1 },							// SWAP Dn
	{ "01001010 11eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_TAS),tas_dn_byte,OP_CYCLES(4,1,0),TBL_DN_1 },				// TAS.B Dn
	{ "01001010 11eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_TAS),tas_ea_byte,OP_CYCLES(10,1,1),TBL_EA_BYTE|TBL_EA_ADDCYCLES },				// TAS.B <ea>
	{ "01001110 0100vvvv", NULL,					DISFUNC(Disass_TRAP),trap,OP_CYCLES(38,4,3),TBL_TRAP_1 },									// TRAP #<vector>
	{ "01001110 01110110", NULL,					DISFUNC(Disass_TRAPV),trapv,OP_CYCLES(34,4,3),NULL },								// TRAPV
	{ "01001010 00eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_TST),tst_dn_byte,OP_CYCLES(4,1,0),TBL_DN_1 },								// TST.B Dn
	{ "01001010 01eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_TST),tst_dn_word,OP_CYCLES(4,1,0),TBL_DN_1 },								// TST.W Dn
	{ "01001010 10eeeeee", EFFADDR_1000000_000_00,	DISFUNC(Disass_TST),tst_dn_long,OP_CYCLES(4,1,0),TBL_DN_1 },								// TST.L Dn
	{ "01001010 00eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_TST),tst_ea_byte,OP_CYCLES(4,1,0),TBL_EA_BYTE|TBL_EA_ADDCYCLES },				// TST.B <ea>
	{ "01001010 01eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_TST),tst_ea_word,OP_CYCLES(4,1,0),TBL_EA_WORD|TBL_EA_ADDCYCLES },				// TST.W <ea>
	{ "01001010 10eeeeee", EFFADDR_0011111_110_00,	DISFUNC(Disass_TST),tst_ea_long,OP_CYCLES(4,1,0),TBL_EA_LONG|TBL_EA_ADDCYCLES },				// TST.L <ea>
	{ "01001110 01011ddd", NULL,					DISFUNC(Disass_UNLK),unlk_an,OP_CYCLES(12,3,0),TBL_AN_1 },								// UNLK An

	{ "1010iiii iiiiiiii", NULL,					NULL,M68000_Line_A_OpCode,OP_CYCLES(0,0,0),NULL },									// Line-A
	{ "1111iiii iiiiiiii", NULL,					NULL,M68000_Line_F_OpCode,OP_CYCLES(0,0,0),NULL },									// Line-F
	{ "10100000 11111111", NULL,					NULL,M68000_Line_A_Trap,OP_CYCLES(0,0,0),NULL },									// Line-A

	{ "00000000 00001000", NULL,					NULL,GemDOS_OpCode,OP_CYCLES(0,0,0),NULL },											// GEMDOS_OPCODE
	{ "00000000 00001001", NULL,					NULL,GemDOS_RunOldOpCode,OP_CYCLES(0,0,0),NULL },									// RUNOLDGEMDOS_OPCODE
	{ "00000000 00001010", NULL,					NULL,TOS_ConnectedDrive_OpCode,OP_CYCLES(0,0,0),NULL },								// CONDRV_OPCODE
	{ "00000000 00001011", NULL,					NULL,M68000_TimerD_OpCode,OP_CYCLES(0,0,0),NULL },									// TIMERD_OPCODE
	{ "00000000 00001100", NULL,					NULL,VDI_OpCode,OP_CYCLES(0,0,0),NULL },											// VDI_OPCODE

	{ "" }	// Term
};

//-----------------------------------------------------------------------
/*
	Scan table entry for string, return bit offset or -ve as cannot find
*/
int Tables_ScanTable(char *pszOpCodeString,char *pszMatchString)
{
	char *pszStringOffset;
	
	pszStringOffset = strstr(pszOpCodeString,pszMatchString);
	if (pszStringOffset)
		return(pszStringOffset - pszOpCodeString);
	else
		return(-1);
}

//-----------------------------------------------------------------------
/*
	Write binary value into string, use subtract method
*/
void Tables_AddBinaryString(char *pszString,int Value,int nBits)
{
	int i,Remainder;
	
	for(i=(nBits-1); i>=0; i--) {
		Remainder = Value-Power2Table[i];

		if (Remainder>=0) {
			*pszString++ = '1';
			Value = Remainder;
		}
		else
			*pszString++ = '0';
	}
}

//-----------------------------------------------------------------------
/*
	Return TRUE is string is all numbers, ie no 'rrr' left etc...
*/
BOOL Tables_IsBinary(char *pszString)
{
	int i;

	for(i=0; i<(int)strlen(pszString); i++) {
		if ( (pszString[i]<'0') || (pszString[i]>'9') )	// Digit within range?
			return(FALSE);							// No, still has values in
	}

	return(TRUE);									// String is all 0's and 1's
}

//-----------------------------------------------------------------------
/*
	Convert binary text string to a decimal integer
*/
int Tables_BinaryToDecimal(char *pszString)
{
	int i,j=0,Value=0;

	for(i=(strlen(pszString)-1); i>=0; i--) {
		if (pszString[i]=='1')
			Value += Power2Table[j];
		j++;
	}

	return(Value);
}

//-----------------------------------------------------------------------
/*
	Add function pointers to tables at index 'OpCode'
	This fills in the decode function table with decode routine, and other values such as registers
	used etc... These values are used by the decode functions to read/write data without the need for
	masking and complicated instructions. As the reading of the 'opcode' will causes a cache-miss
	anyway, having such a large decode table does not impeed performance.
*/
void Tables_AddEntryToDecodeTable(char *pszOpCodeString)
{
	unsigned long Data;
	int OpCode,OpCodeIndex;

	// Convert string to decimal index into table
	OpCode = Tables_BinaryToDecimal(pszOpCodeString);
	OpCodeIndex = STMemory_Swap68000Int(OpCode)*SIZEOF_DECODE;

	// Do simple error check as entry in table SHOULD be NULL
#ifdef USE_DEBUGGER
	if (DecodeTable[OpCodeIndex+(DECODE_DISASS/sizeof(long))]) {
		Debug_File("ERROR! OpCode table entry already found 0x%X\n",pszOpCodeString);
	}
#endif

	// Add disassembly and decode functions to table look up
	DecodeTable[OpCodeIndex+(DECODE_FUNCTION/sizeof(long))] = (unsigned long)TablePopulate[TableIndex].pDecodeFunction;	// Decode function
	DecodeTable[OpCodeIndex+(DECODE_CYCLES/sizeof(long))] = (unsigned long)TablePopulate[TableIndex].Cycles;			// Instruction cycle time

	// <ea>
	if ( (TablePopulate[TableIndex].Size&TBL_EA_MASK)==TBL_EA_BYTE ) {
		DecodeTable[OpCodeIndex+(DECODE_EA/sizeof(long))] = EffAddrFunctions_Byte[OpCode&0x3f];						// <ea> function
		if ( TablePopulate[TableIndex].Size&TBL_EA_ADDCYCLES )														// + <ea> time
			DecodeTable[OpCodeIndex+(DECODE_CYCLES/sizeof(long))] += EffAddrFunctions_ByteCycles[OpCode&0x3f];
	}
	if ( (TablePopulate[TableIndex].Size&TBL_EA_MASK)==TBL_EA_WORD ) {
		DecodeTable[OpCodeIndex+(DECODE_EA/sizeof(long))] = EffAddrFunctions_Word[OpCode&0x3f];
		if ( TablePopulate[TableIndex].Size&TBL_EA_ADDCYCLES )														// + <ea> time
			DecodeTable[OpCodeIndex+(DECODE_CYCLES/sizeof(long))] += EffAddrFunctions_WordCycles[OpCode&0x3f];
	}
	if ( (TablePopulate[TableIndex].Size&TBL_EA_MASK)==TBL_EA_LONG ) {
		DecodeTable[OpCodeIndex+(DECODE_EA/sizeof(long))] = EffAddrFunctions_Long[OpCode&0x3f];
		if ( TablePopulate[TableIndex].Size&TBL_EA_ADDCYCLES )														// + <ea> time
			DecodeTable[OpCodeIndex+(DECODE_CYCLES/sizeof(long))] += EffAddrFunctions_LongCycles[OpCode&0x3f];
	}
	if ( (TablePopulate[TableIndex].Size&TBL_EA_MASK)==TBL_EA_LOAD ) {
		DecodeTable[OpCodeIndex+(DECODE_EA/sizeof(long))] = LoadEffAddrFunctions[OpCode&0x3f];
		if (TablePopulate[TableIndex].Size&TBL_EA_ADDCYCLES)														// + <ea> time
			DecodeTable[OpCodeIndex+(DECODE_CYCLES/sizeof(long))] += LoadEffAddrFunctions_LongCycles[OpCode&0x3f];
	}
	// Set 'An' as data or<ea>
	DecodeTable[OpCodeIndex+(DECODE_1/sizeof(long))] = (unsigned long)&Regs[REG_A0+(OpCode&0x7)];					// <ea> => An

	// Data_2
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_2)==TBL_DATA_2 ) {
		Data = (unsigned long)((OpCode>>9)&0x7);																	// Data 0..7 (0 is '8')
		if (Data==0)
			Data = 8;
		DecodeTable[OpCodeIndex+(DECODE_2/sizeof(long))] = Data;
	}
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_1)==TBL_DN_1 )
		DecodeTable[OpCodeIndex+(DECODE_1/sizeof(long))] = (unsigned long)&Regs[OpCode&0x7];						// DReg
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_1)==TBL_DNUPR_1 )
		DecodeTable[OpCodeIndex+(DECODE_1/sizeof(long))] = (unsigned long)&Regs[(OpCode>>9)&0x7];					// DRegUpper
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_1)==TBL_AN_1 )
		DecodeTable[OpCodeIndex+(DECODE_1/sizeof(long))] = (unsigned long)&Regs[REG_A0+(OpCode&0x7)];				// AReg
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_1)==TBL_ANUPR_1 )
		DecodeTable[OpCodeIndex+(DECODE_1/sizeof(long))] = (unsigned long)&Regs[REG_A0+((OpCode>>9)&0x7)];			// ARegUpper
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_1)==TBL_SOFFSET_1 )
		DecodeTable[OpCodeIndex+(DECODE_1/sizeof(long))] = (unsigned long)((long)((char)OpCode));					// 32-bit signed offset of lower 8 bits
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_1)==TBL_TRAP_1 )
		DecodeTable[OpCodeIndex+(DECODE_1/sizeof(long))] = ((OpCode&0xf)*4)+EXCEPTION_TRAP0;						// Trap Vector

	if ( (TablePopulate[TableIndex].Size&TBL_MASK_2)==TBL_DN_2 )
		DecodeTable[OpCodeIndex+(DECODE_2/sizeof(long))] = (unsigned long)&Regs[OpCode&0x7];						// DReg
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_2)==TBL_DNUPR_2 )
		DecodeTable[OpCodeIndex+(DECODE_2/sizeof(long))] = (unsigned long)&Regs[(OpCode>>9)&0x7];					// DRegUpper
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_2)==TBL_AN_2 )
		DecodeTable[OpCodeIndex+(DECODE_2/sizeof(long))] = (unsigned long)&Regs[REG_A0+(OpCode&0x7)];				// AReg
	if ( (TablePopulate[TableIndex].Size&TBL_MASK_2)==TBL_ANUPR_2 )
		DecodeTable[OpCodeIndex+(DECODE_2/sizeof(long))] = (unsigned long)&Regs[REG_A0+((OpCode>>9)&0x7)];			// ARegUpper


#ifdef USE_DEBUGGER
	DecodeTable[OpCodeIndex+(DECODE_DISASS/sizeof(long))] = (unsigned long)TablePopulate[TableIndex].pDisassFunction;
#endif

	// Round cycle times up to multiple of 4 for an ST - ie 10 becomes 12. Sounds good!
	DecodeTable[OpCodeIndex+(DECODE_CYCLES/sizeof(long))] = ROUND_CYCLES_TO4( DecodeTable[OpCodeIndex+(DECODE_CYCLES/sizeof(long))] );
}

//-----------------------------------------------------------------------
/*
	Recursive function to add entries to decode tables
*/
void Tables_AddCode(char *pszOpCodeString,int Index)
{
	char szNewString[16+1];
	int i,Start,BitOffset;

	// Recurse add
	if (pszScanStrings[Index]) {
		// Find if sub string is found in OpCode
		BitOffset = Tables_ScanTable(pszOpCodeString,pszScanStrings[Index]);
		if (BitOffset>=0) {
			// Replace entries in 'pszOpCodeString' with numbers and recurse
			strcpy(szNewString,pszOpCodeString);
			// Find starting index
			Start = 0;
			if (!strcmp(pszScanStrings[Index],"CCCC"))
				Start = 2;
			else if (!strcmp(pszScanStrings[Index],"ffffffff"))
				Start = 1;

			for(i=Start; i<Power2Table[strlen(pszScanStrings[Index])]; i++) {
				// Add chars for binary string for value 'i'
				Tables_AddBinaryString(&szNewString[BitOffset],i,strlen(pszScanStrings[Index]));
				if (Tables_IsBinary(szNewString))
					Tables_AddEntryToDecodeTable(szNewString);
				else
					Tables_AddCode(szNewString,Index+1);
			}
		}
		else {
			if (!Tables_IsBinary(pszOpCodeString))
				Tables_AddCode(pszOpCodeString,Index+1);
			else
				Tables_AddEntryToDecodeTable(pszOpCodeString);
		}
	}
}

//-----------------------------------------------------------------------
/*
	Replace 'eeeeee' in instruction define with string, ie '000RRR' for say Dn
*/
void Tables_ReplaceEffAddr(char *pszString,char *pszEffString)
{
	int i;

	for(i=0; i<6; i++)
		pszString[i+10] = pszEffString[i];	// Write over last 6 characters
}

//-----------------------------------------------------------------------
/*
	Add Effective Address codes to tables
*/
void Tables_AddCodeEffAddr(char *pszOpCodeString,int EffAddr)
{
	// Check each effective address, replace 'eeeeee' in string and add code
	if ((EffAddr>>11)&1) {	// Dn
		Tables_ReplaceEffAddr(pszOpCodeString,"000RRR");
		Tables_AddCode(pszOpCodeString,0);
	}
	if ((EffAddr>>10)&1) {	// An
		Tables_ReplaceEffAddr(pszOpCodeString,"001RRR");
		Tables_AddCode(pszOpCodeString,0);
	}
	if ((EffAddr>>9)&1) {	// (An)
		Tables_ReplaceEffAddr(pszOpCodeString,"010RRR");
		Tables_AddCode(pszOpCodeString,0);
	}
	if ((EffAddr>>8)&1) {	// (An)+
		Tables_ReplaceEffAddr(pszOpCodeString,"011RRR");
		Tables_AddCode(pszOpCodeString,0);
	}
	if ((EffAddr>>7)&1) {	// -(An)
		Tables_ReplaceEffAddr(pszOpCodeString,"100RRR");
		Tables_AddCode(pszOpCodeString,0);
	}
	if ((EffAddr>>6)&1) {	// (d16,An)
		Tables_ReplaceEffAddr(pszOpCodeString,"101RRR");
		Tables_AddCode(pszOpCodeString,0);
	}
	if ((EffAddr>>5)&1) {	// (d8,An,Xn)
		Tables_ReplaceEffAddr(pszOpCodeString,"110RRR");
		Tables_AddCode(pszOpCodeString,0);
	}

	if ((EffAddr>>4)&1) {	// (xxx).W
		Tables_ReplaceEffAddr(pszOpCodeString,"111000");
		Tables_AddCode(pszOpCodeString,0);
	}
	if ((EffAddr>>3)&1) {	// (xxx).L
		Tables_ReplaceEffAddr(pszOpCodeString,"111001");
		Tables_AddCode(pszOpCodeString,0);
	}
	if ((EffAddr>>2)&1) {	// #<data>
		Tables_ReplaceEffAddr(pszOpCodeString,"111100");
		Tables_AddCode(pszOpCodeString,0);
	}

	if ((EffAddr>>1)&1) {	// (d16,PC)
		Tables_ReplaceEffAddr(pszOpCodeString,"111010");
		Tables_AddCode(pszOpCodeString,0);
	}
	if (EffAddr&1) {		// (d8,PC,Xn)
		Tables_ReplaceEffAddr(pszOpCodeString,"111011");
		Tables_AddCode(pszOpCodeString,0);
	}
}

//-----------------------------------------------------------------------
/*
	Create decode tables
*/
void Tables_SetupDecodeTable(void)
{
	char szString[16+1];
	int i;

	// Clear tables, set default as illegal instruction
	Memory_Clear(DecodeTable,sizeof(unsigned long)*(65536*SIZEOF_DECODE));
	for(i=0; i<65536; i++)
		DecodeTable[(i*SIZEOF_DECODE)+(DECODE_FUNCTION/sizeof(long))] = (unsigned long)M68000_IllegalInstruction;

	// Read each instruction
	TableIndex = 0;
	while(strlen(TablePopulate[TableIndex].szOpCode)>0) {
		// Remove any spaces, easier to read
		Misc_RemoveSpacesFromString(TablePopulate[TableIndex].szOpCode,szString);

		// Does 'OpCode' have effective address?
		if (strstr(szString,"eeeeee"))
			Tables_AddCodeEffAddr(szString,TablePopulate[TableIndex].EffAddr);
		else
			Tables_AddCode(szString,0);

		TableIndex++;
	}
}

//-----------------------------------------------------------------------
/*
	Setup Effective Address function tables
	NOTE - MUST be executed BEFORE Tables_SetupDecodeTable()
*/
void Tables_SetupEffAddrTables(void)
{
	int i;

	// And our second set of tables!!
	Memory_Clear(EffAddrFunctions_Byte,sizeof(unsigned long)*64);
	Memory_Clear(EffAddrFunctions_Word,sizeof(unsigned long)*64);
	Memory_Clear(EffAddrFunctions_Long,sizeof(unsigned long)*64);
	Memory_Clear(LoadEffAddrFunctions,sizeof(unsigned long)*64);
	Memory_Clear(EffAddrFunctions_ByteCycles,sizeof(unsigned long)*64);
	Memory_Clear(EffAddrFunctions_WordCycles,sizeof(unsigned long)*64);
	Memory_Clear(EffAddrFunctions_LongCycles,sizeof(unsigned long)*64);
	Memory_Clear(LoadEffAddrFunctions_LongCycles,sizeof(unsigned long)*64);

	//---------------------------------------------------------------------------------
	// Dn
	for(i=0; i<8; i++) {
		EffAddrFunctions_Byte[ BIN6(0,0,0, 0,0,0) + i ] = EffAddrTable_Dn[i];
		EffAddrFunctions_Word[ BIN6(0,0,0, 0,0,0) + i ] = EffAddrTable_Dn[i];
		EffAddrFunctions_Long[ BIN6(0,0,0, 0,0,0) + i ] = EffAddrTable_Dn[i];
	}

	//---------------------------------------------------------------------------------
	// An
	for(i=0; i<8; i++) {
		EffAddrFunctions_Byte[ BIN6(0,0,1, 0,0,0) + i ] = EffAddrTable_An[i];
		EffAddrFunctions_Word[ BIN6(0,0,1, 0,0,0) + i ] = EffAddrTable_An[i];
		EffAddrFunctions_Long[ BIN6(0,0,1, 0,0,0) + i ] = EffAddrTable_An[i];
	}

	//---------------------------------------------------------------------------------
	// (An)
	for(i=0; i<8; i++) {
		EffAddrFunctions_Byte[ BIN6(0,1,0, 0,0,0) + i ] = (unsigned long)effaddr_read_010_byte;
		EffAddrFunctions_Word[ BIN6(0,1,0, 0,0,0) + i ] = (unsigned long)effaddr_read_010_word;
		EffAddrFunctions_Long[ BIN6(0,1,0, 0,0,0) + i ] = (unsigned long)effaddr_read_010_long;
		LoadEffAddrFunctions[ BIN6(0,1,0, 0,0,0) + i ] = (unsigned long)loadeffaddr_010;
		EffAddrFunctions_ByteCycles[ BIN6(0,1,0, 0,0,0) + i ] = OP_CYCLES(4,1,0);
		EffAddrFunctions_WordCycles[ BIN6(0,1,0, 0,0,0) + i ] = OP_CYCLES(4,1,0);
		EffAddrFunctions_LongCycles[ BIN6(0,1,0, 0,0,0) + i ] = OP_CYCLES(8,2,0);
		LoadEffAddrFunctions_LongCycles[ BIN6(0,1,0, 0,0,0) + i ] = OP_CYCLES(4,1,0);
	}

	//---------------------------------------------------------------------------------
	// (An)+
	for(i=0; i<8; i++) {
		EffAddrFunctions_Byte[ BIN6(0,1,1, 0,0,0) + i ] = (unsigned long)effaddr_read_011_byte;
		EffAddrFunctions_Word[ BIN6(0,1,1, 0,0,0) + i ] = (unsigned long)effaddr_read_011_word;
		EffAddrFunctions_Long[ BIN6(0,1,1, 0,0,0) + i ] = (unsigned long)effaddr_read_011_long;
		EffAddrFunctions_ByteCycles[ BIN6(0,1,1, 0,0,0) + i ] = OP_CYCLES(4,1,0);
		EffAddrFunctions_WordCycles[ BIN6(0,1,1, 0,0,0) + i ] = OP_CYCLES(4,1,0);
		EffAddrFunctions_LongCycles[ BIN6(0,1,1, 0,0,0) + i ] = OP_CYCLES(8,2,0);
	}
	// (A7)+, needs special case
	EffAddrFunctions_Byte[ BIN6(0,1,1, 0,0,0) + 7 ] = (unsigned long)effaddr_read_011_byte_A7;

	//---------------------------------------------------------------------------------
	// -(An)
	for(i=0; i<8; i++) {
		EffAddrFunctions_Byte[ BIN6(1,0,0, 0,0,0) + i ] = (unsigned long)effaddr_read_100_byte;
		EffAddrFunctions_Word[ BIN6(1,0,0, 0,0,0) + i ] = (unsigned long)effaddr_read_100_word;
		EffAddrFunctions_Long[ BIN6(1,0,0, 0,0,0) + i ] = (unsigned long)effaddr_read_100_long;
		EffAddrFunctions_ByteCycles[ BIN6(1,0,0, 0,0,0) + i ] = OP_CYCLES(6,1,0);	//OP_CYCLES(4,1,0);
		EffAddrFunctions_WordCycles[ BIN6(1,0,0, 0,0,0) + i ] = OP_CYCLES(6,1,0);	//OP_CYCLES(4,1,0);
		EffAddrFunctions_LongCycles[ BIN6(1,0,0, 0,0,0) + i ] = OP_CYCLES(10,2,0);	//OP_CYCLES(8,2,0);
	}
	// -(A7), needs special case
	EffAddrFunctions_Byte[ BIN6(1,0,0, 0,0,0) + 7 ] = (unsigned long)effaddr_read_100_byte_A7;

	//---------------------------------------------------------------------------------
	// (d16,An)
	for(i=0; i<8; i++) {
		EffAddrFunctions_Byte[ BIN6(1,0,1, 0,0,0) + i ] = (unsigned long)effaddr_read_101_byte;
		EffAddrFunctions_Word[ BIN6(1,0,1, 0,0,0) + i ] = (unsigned long)effaddr_read_101_word;
		EffAddrFunctions_Long[ BIN6(1,0,1, 0,0,0) + i ] = (unsigned long)effaddr_read_101_long;
		LoadEffAddrFunctions[ BIN6(1,0,1, 0,0,0) + i ] = (unsigned long)loadeffaddr_101;
		EffAddrFunctions_ByteCycles[ BIN6(1,0,1, 0,0,0) + i ] = OP_CYCLES(8,2,0);
		EffAddrFunctions_WordCycles[ BIN6(1,0,1, 0,0,0) + i ] = OP_CYCLES(8,2,0);
		EffAddrFunctions_LongCycles[ BIN6(1,0,1, 0,0,0) + i ] = OP_CYCLES(12,3,0);
		LoadEffAddrFunctions_LongCycles[ BIN6(1,0,1, 0,0,0) + i ] = OP_CYCLES(8,2,0);
	}

	//---------------------------------------------------------------------------------
	// (d8,An,Xn)
	for(i=0; i<8; i++) {
		EffAddrFunctions_Byte[ BIN6(1,1,0, 0,0,0) + i ] = (unsigned long)effaddr_read_110_byte;
		EffAddrFunctions_Word[ BIN6(1,1,0, 0,0,0) + i ] = (unsigned long)effaddr_read_110_word;
		EffAddrFunctions_Long[ BIN6(1,1,0, 0,0,0) + i ] = (unsigned long)effaddr_read_110_long;
		LoadEffAddrFunctions[ BIN6(1,1,0, 0,0,0) + i ] = (unsigned long)loadeffaddr_110;
		EffAddrFunctions_ByteCycles[ BIN6(1,1,0, 0,0,0) + i ] = OP_CYCLES(10,2,0);
		EffAddrFunctions_WordCycles[ BIN6(1,1,0, 0,0,0) + i ] = OP_CYCLES(10,2,0);
		EffAddrFunctions_LongCycles[ BIN6(1,1,0, 0,0,0) + i ] = OP_CYCLES(14,3,0);
		LoadEffAddrFunctions_LongCycles[ BIN6(1,1,0, 0,0,0) + i ] = OP_CYCLES(12,2,0);
	}

	//---------------------------------------------------------------------------------
	// (xxx).W
	EffAddrFunctions_Byte[ BIN6(1,1,1, 0,0,0) ] = (unsigned long)effaddr_read_111_000_byte;
	EffAddrFunctions_Word[ BIN6(1,1,1, 0,0,0) ] = (unsigned long)effaddr_read_111_000_word;
	EffAddrFunctions_Long[ BIN6(1,1,1, 0,0,0) ] = (unsigned long)effaddr_read_111_000_long;
	LoadEffAddrFunctions[ BIN6(1,1,1, 0,0,0) ] = (unsigned long)loadeffaddr_111_000;
	EffAddrFunctions_ByteCycles[ BIN6(1,1,1, 0,0,0) ] = OP_CYCLES(8,2,0);
	EffAddrFunctions_WordCycles[ BIN6(1,1,1, 0,0,0) ] = OP_CYCLES(8,2,0);
	EffAddrFunctions_LongCycles[ BIN6(1,1,1, 0,0,0) ] = OP_CYCLES(12,3,0);
	LoadEffAddrFunctions_LongCycles[ BIN6(1,1,1, 0,0,0) ] = OP_CYCLES(8,2,0);

	//---------------------------------------------------------------------------------
	// (xxx).L
	EffAddrFunctions_Byte[ BIN6(1,1,1, 0,0,1) ] = (unsigned long)effaddr_read_111_001_byte;
	EffAddrFunctions_Word[ BIN6(1,1,1, 0,0,1) ] = (unsigned long)effaddr_read_111_001_word;
	EffAddrFunctions_Long[ BIN6(1,1,1, 0,0,1) ] = (unsigned long)effaddr_read_111_001_long;
	LoadEffAddrFunctions[ BIN6(1,1,1, 0,0,1) ] = (unsigned long)loadeffaddr_111_001;
	EffAddrFunctions_ByteCycles[ BIN6(1,1,1, 0,0,1) ] = OP_CYCLES(12,3,0);
	EffAddrFunctions_WordCycles[ BIN6(1,1,1, 0,0,1) ] = OP_CYCLES(12,3,0);
	EffAddrFunctions_LongCycles[ BIN6(1,1,1, 0,0,1) ] = OP_CYCLES(16,4,0);
	LoadEffAddrFunctions_LongCycles[ BIN6(1,1,1, 0,0,1) ] = OP_CYCLES(12,3,0);

	//---------------------------------------------------------------------------------
	// #<data>
	EffAddrFunctions_Byte[ BIN6(1,1,1, 1,0,0) ] = (unsigned long)effaddr_read_111_100_byte;
	EffAddrFunctions_Word[ BIN6(1,1,1, 1,0,0) ] = (unsigned long)effaddr_read_111_100_word;
	EffAddrFunctions_Long[ BIN6(1,1,1, 1,0,0) ] = (unsigned long)effaddr_read_111_100_long;
	EffAddrFunctions_ByteCycles[ BIN6(1,1,1, 1,0,0) ] = OP_CYCLES(4,1,0);
	EffAddrFunctions_WordCycles[ BIN6(1,1,1, 1,0,0) ] = OP_CYCLES(4,1,0);
	EffAddrFunctions_LongCycles[ BIN6(1,1,1, 1,0,0) ] = OP_CYCLES(8,2,0);

	//---------------------------------------------------------------------------------
	// (d16,PC)
	EffAddrFunctions_Byte[ BIN6(1,1,1, 0,1,0) ] = (unsigned long)effaddr_read_111_010_byte;
	EffAddrFunctions_Word[ BIN6(1,1,1, 0,1,0) ] = (unsigned long)effaddr_read_111_010_word;
	EffAddrFunctions_Long[ BIN6(1,1,1, 0,1,0) ] = (unsigned long)effaddr_read_111_010_long;
	LoadEffAddrFunctions[ BIN6(1,1,1, 0,1,0) ] = (unsigned long)loadeffaddr_111_010;
	EffAddrFunctions_ByteCycles[ BIN6(1,1,1, 0,1,0) ] = OP_CYCLES(8,2,0);
	EffAddrFunctions_WordCycles[ BIN6(1,1,1, 0,1,0) ] = OP_CYCLES(8,2,0);
	EffAddrFunctions_LongCycles[ BIN6(1,1,1, 0,1,0) ] = OP_CYCLES(12,3,0);
	LoadEffAddrFunctions_LongCycles[ BIN6(1,1,1, 0,1,0) ] = OP_CYCLES(8,2,0);

	//---------------------------------------------------------------------------------
	// (d8,PC,Xn)
	EffAddrFunctions_Byte[ BIN6(1,1,1, 0,1,1) ] = (unsigned long)effaddr_read_111_011_byte;
	EffAddrFunctions_Word[ BIN6(1,1,1, 0,1,1) ] = (unsigned long)effaddr_read_111_011_word;
	EffAddrFunctions_Long[ BIN6(1,1,1, 0,1,1) ] = (unsigned long)effaddr_read_111_011_long;
	LoadEffAddrFunctions[ BIN6(1,1,1, 0,1,1) ] = (unsigned long)loadeffaddr_111_011;
	EffAddrFunctions_ByteCycles[ BIN6(1,1,1, 0,1,1) ] = OP_CYCLES(10,2,0);
	EffAddrFunctions_WordCycles[ BIN6(1,1,1, 0,1,1) ] = OP_CYCLES(10,2,0);
	EffAddrFunctions_LongCycles[ BIN6(1,1,1, 0,1,1) ] = OP_CYCLES(14,3,0);
	LoadEffAddrFunctions_LongCycles[ BIN6(1,1,1, 0,1,1) ] = OP_CYCLES(12,2,0);
}

//-----------------------------------------------------------------------
/*
	Create condition code table

	PC-FLAGS are :-

	0: carry
	1: 1
	2: parity
	3: 0
	4: auxiliary carry
	5: 0 
	6: zero
	7: sign
	8: trap
	9: interrupt
	10: direction
	11: overflow
	12: i/o privilege level
	13: i/o privilege level
	14: nested
	15: 0
	16: resume flag
	17: virtual 8086 mode
	18: alignment check
	19-31	0

	auxiliary carry = carry out of bit position 3
	parity = low byte of result has even no. of bits set
*/
void Tables_SetupConditionCodeTable(void)
{
	unsigned char CCode;
	int i;

	// PC codition codes to NZV, CX are same(taken from carry)
	for(i=0; i<4096; i++) {
		CCode = 0;

		if (i&PC_CARRY)		CCode |= SR_CARRY | SR_AUX;
		if (i&PC_ZERO)		CCode |= SR_ZERO;
		if (i&PC_NEG)		CCode |= SR_NEG;
		if (i&PC_OVERFLOW)	CCode |= SR_OVERFLOW;

		PCCodeTable_NZV_CX_SAME[i] = CCode;
	}

	// PC codition codes to NZVC
	for(i=0; i<4096; i++) {
		CCode = 0;

		if (i&PC_CARRY)		CCode |= EMU_C;
		if (i&PC_ZERO)		CCode |= EMU_Z;
		if (i&PC_NEG)		CCode |= EMU_N;
		if (i&PC_OVERFLOW)	CCode |= EMU_V;

		PCCodeTable_NZVC[i] = CCode;
	}
}

//-----------------------------------------------------------------------
/*
	Setup shift cycles table(as rounds up to nearest 4). Use with ASL/R, LSL/R, ROL/R and ROXL/ROXR
*/
void Tables_SetupShiftCycleTable(void)
{
	int i;

	// Set up table for shift cycle count, byte/word is 6+2n and long is 8+2n
	for(i=0; i<64; i++) {
		ShiftCycleTable_ByteWord[i] = ROUND_CYCLES_TO4( 6+(2*i) );
		ShiftCycleTable_Long[i] = ROUND_CYCLES_TO4( 8+(2*i) );
	}
}

//-----------------------------------------------------------------------
/*
	Setup all 68000 decode tables, complete with cycle times and condition code conversions
*/
void Tables_Init(void)
{
	// Setup decode/debugger functions and cycle times
	Tables_SetupEffAddrTables();
	Tables_SetupDecodeTable();
	Tables_SetupConditionCodeTable();
	Tables_SetupShiftCycleTable();
}
