
#include <iostream>
#include "Intellivision.h"

using namespace std;

Intellivision::Intellivision()
    : player1Controller(PLAYER_ONE),
      player2Controller(PLAYER_TWO),
      psg(0x01F0, &player1Controller, &player2Controller),
      RAM8bit(0x00F0, 0x0100, 8),
      RAM16bit(0x0070, 0x02F0, 16),
      execROM(0x1000, 0x1000)
{
    intellivoiceInUse = FALSE;
    ecsInUse = FALSE;
	currentCartridge = NULL;
    init();
}

Intellivision::~Intellivision()
{
    //delete the audio output lines
    audioMixer.releaseLine(&psgAudioLine);
    audioMixer.releaseLine(&ecsAudioLine);
    audioMixer.releaseLine(&ivoiceAudioLine);
}

#ifdef __cplusplus
extern "C" {
#endif

void xbox_set_memory ( char *memptr ) ;

#ifdef __cplusplus
}
#endif

/**
 * Initializes all of the basic hardware included in the Intellivision
 * Master Component as well as the ECS and Intellivoice peripherals.
 * This method is called only once when the Intellivision is constructed.
 */
void Intellivision::init()
{
    //create the hand controllers
    controllerBus.addController(&player1Controller);
    controllerBus.addController(&player2Controller);

    //8 bit RAM used by EXEC and cartridge
    memoryBus.addMemory(&RAM8bit);

	//xbox_set_memory( (char*)RAM8bit.getmemory() ) ;

    //the executive (EXEC) rom
    memoryBus.addMemory(&execROM);

    //16 bit RAM used for...
    //    02F0       EXEC Object GRAM Allocation Table Pointer
    //    02F1-0318  1610 Stack
    //    0319-031C  16 bit RAM for EXEC
    //    031D-035C  Moving Object RAM Data Bases
    //    035D-035F  16 bit RAM for EXEC
    memoryBus.addMemory(&RAM16bit);

    //init the CPU
    cpu.init(&memoryBus, &SR1toINTRMLine, &SR2toBUSRQLine,
            &BUSAKtoSSTLine, 0x1000, 0x1004);
    processorBus.addProcessor(&cpu);

    //init the STIC
    stic.init(&cpu, &SR1toINTRMLine, &SR2toBUSRQLine,
            &BUSAKtoSSTLine);
    setVideoProducer(&stic);
    processorBus.addProcessor(&stic);
    INT32 memoryCount = stic.getMemoryCount();
    for (INT32 i = 0; i < memoryCount; i++)
        memoryBus.addMemory(stic.getMemory(i));

    //init the sound chip
    audioMixer.createLine(&psgAudioLine);
    psg.setAudioOutputLine(psgAudioLine);
    psgAudioLine->acquire();
    processorBus.addProcessor(&psg);
    memoryBus.addMemory(&psg.registers);

    //init the ECS for ECS cartridge support
    audioMixer.createLine(&ecsAudioLine);
    this->ecs.psg2.setAudioOutputLine(ecsAudioLine);

    //init the Intellivoice for Intellivoice cartridge support
    audioMixer.createLine(&ivoiceAudioLine);
    this->intellivoice.setAudioOutputLine(ivoiceAudioLine);
}

/**
 * Pushes the reset button on the Intellivision.
 */
void Intellivision::reset()
{
    Emulator::reset();
    memoryBus.reset();
}
#ifdef __cplusplus
extern "C" {
#endif

void writexbox( char *msg) ;


#ifdef __cplusplus
}
#endif

/**
 * Inserts a cartridge into the Intellivision cartridge slot.
 *
 * @param cartridge the cartridge to insert
 */
void Intellivision::insertCartridge(IntellivisionCartridge* cartridge)
{
	writexbox("cart") ;
    //remove the old cartridge
    IntellivisionCartridge* oldCart = currentCartridge;
	writexbox("cart") ;
    if (currentCartridge != NULL)
	{
	writexbox("cartx") ;
        removeCartridge();
	writexbox("cartx") ;
	}
	writexbox("cart") ;
    this->currentCartridge = cartridge;

    //add the ECS if this cartridge needs it
	writexbox("cart") ;
    if (currentCartridge->requiresECS()) {
	writexbox("cartyy") ;
        processorBus.addProcessor(&ecs.psg2);
        memoryBus.addMemory(&ecs.psg2.registers);
        memoryBus.addMemory(&ecs.bank0);
        memoryBus.addMemory(&ecs.bank1);
        memoryBus.addMemory(&ecs.bank2);
        memoryBus.addMemory(&ecs.ramBank);
        memoryBus.addMemory(&ecs.uart);
        ecsAudioLine->acquire();
        ecsInUse = TRUE;
    }

	writexbox("cart") ;
    //add the Intellivoice if this cartridge can use it and we
    //have the ROM necessary for it
    if (currentCartridge->usesIntellivoice() &&
            intellivoice.hasIntellivoiceImage())
    {
	writexbox("cartzz") ;
        processorBus.addProcessor(&intellivoice.microSequencer);
        processorBus.addProcessor(&intellivoice.lpc12);
        memoryBus.addMemory(&intellivoice.microSequencer.registers);
        ivoiceAudioLine->acquire();
        intellivoiceInUse = TRUE;
    }

	writexbox("cart") ;
    //add the cartridge
    INT32 memoryCount = cartridge->getMemoryCount();
	writexbox("cart") ;
    for (INT32 i = 0; i < memoryCount; i++)
        memoryBus.addMemory(cartridge->getMemory(i));
	writexbox("cart") ;
}

IntellivisionCartridge* Intellivision::getCartridge()
{
    return currentCartridge;
}

/**
 * Removes the current cartridge from the Intellivision cartridge slot.
 */
void Intellivision::removeCartridge()
{
    if (currentCartridge == NULL)
        return;

    IntellivisionCartridge* oldCart = currentCartridge;
    //remove the cartridge
    INT32 memoryCount = currentCartridge->getMemoryCount();
    for (INT32 i = 0; i < memoryCount; i++)
        memoryBus.removeMemory((Memory16Bit*)currentCartridge->getMemory(i));

    //remove the Intellivoice
    if (intellivoiceInUse) {
        ivoiceAudioLine->release();
        memoryBus.removeMemory(&intellivoice.microSequencer.registers);
        processorBus.removeProcessor(&intellivoice.lpc12);
        processorBus.removeProcessor(&intellivoice.microSequencer);
        intellivoiceInUse = FALSE;
    }

    if (ecsInUse) {
        ecsAudioLine->release();
        memoryBus.removeMemory(&ecs.uart);
        memoryBus.removeMemory(&ecs.ramBank);
        memoryBus.removeMemory(&ecs.bank2);
        memoryBus.removeMemory(&ecs.bank1);
        memoryBus.removeMemory(&ecs.bank0);
        memoryBus.removeMemory(&ecs.psg2.registers);
        processorBus.removeProcessor(&ecs.psg2);
        ecsInUse = TRUE;
    }

    currentCartridge = NULL;
}

void Intellivision::setExecImage(UINT16* execImage)
{
    execROM.loadImage(execImage);
}

void Intellivision::setGROMImage(UINT16* gromImage)
{
    stic.setGROMImage(gromImage);
}

void Intellivision::enableIntellivoiceSupport(UINT16* image)
{
    intellivoice.setIntellivoiceImage(image);
}

void Intellivision::enableECSSupport(UINT16* image)
{
    ecs.setECSROMImage(image);
}


