#define  INITGUID

#include <windows.h>
#include "shared.h"
#include "ddraw.h"
#include "d3d.h"
#include "dinput.h"
#include "main.h"
#include "input.h"
#include "registry.h"
#include "modes.h"

static LPDIRECTDRAW7						pDirectDraw		= NULL;
static LPDIRECTDRAWSURFACE7					pFrontSurface	= NULL;
static LPDIRECTDRAWSURFACE7					pBackSurface	= NULL;
static LPDIRECT3D7							pDirect3D		= NULL;
static LPDIRECT3DDEVICE7					p3DDevice		= NULL;
static LPDIRECTDRAWSURFACE7					pSystemTexture  = NULL;
static LPDIRECTDRAWSURFACE7					pVRAMTexture	= NULL;
static LPDIRECTDRAWPALETTE					pPalette		= NULL;
static LPDIRECTDRAWCLIPPER					pClipper		= NULL;
static MODE									Mode[MAX_MODES];
static int									NoOfModes;
static int									CurrentMode		= -1;
static BOOL									ModeIsFullScreen = FALSE;
static RECT									ViewportRect;
static RECT									ScreenRect;
static D3DTLVERTEX							Vertex[4];
static D3DTLVERTEX							ScanLine[2];
static BOOL									UseGDISurface = FALSE;
static HMODULE								hModule		  = NULL;

BOOL InitModes(void)
{
	int				i;
	int				Devices = 0;
	HRESULT		    (WINAPI *pDirectDrawCreateEx)(GUID FAR *, LPVOID *, REFIID, IUnknown FAR *);

	ModeIsFullScreen = FALSE;

	hModule = LoadLibrary("ddraw.dll");

	if (!hModule) 
	{
		FreeLibrary(hModule);
		hModule = NULL;
	
		return FALSE;
	}

	pDirectDrawCreateEx = (void *)GetProcAddress(hModule, "DirectDrawCreateEx");

	if (!pDirectDrawCreateEx) return FALSE;

	if (pDirectDrawCreateEx(NULL, &pDirectDraw, &IID_IDirectDraw7, NULL) != DD_OK) return FALSE;

	memset(Mode, 0, sizeof(MODE) * MAX_MODES);
	NoOfModes = 0;
	IDirectDraw7_EnumDisplayModes(pDirectDraw, DDEDM_REFRESHRATES, NULL, NULL, EnumModes);

	if (IDirectDraw7_QueryInterface(pDirectDraw, &IID_IDirect3D7, &pDirect3D) != DD_OK)
	{
		TidyModes();

		return FALSE;
	}

	memset(&Vertex, 0, sizeof(D3DTLVERTEX) * 4);
	memset(&ScanLine, 0, sizeof(D3DTLVERTEX) * 2);
	
	for (i = 0 ; i < 4 ; i++)
	{
		Vertex[i].rhw   = 1;
		Vertex[i].color	= 0xFFFFFFFF;
	}
	
	return TRUE;
}

HRESULT WINAPI EnumModes(LPDDSURFACEDESC2 pDesc, LPVOID pContext)
{
	if (pDesc->ddpfPixelFormat.dwRGBBitCount != 8 && pDesc->ddpfPixelFormat.dwRGBBitCount != 24)
	{
		Mode[NoOfModes].Width		= pDesc->dwWidth;
		Mode[NoOfModes].Height		= pDesc->dwHeight;
		Mode[NoOfModes].BitDepth	= pDesc->ddpfPixelFormat.dwRGBBitCount;

		NoOfModes++;

		if (NoOfModes == MAX_MODES) return DDENUMRET_CANCEL;
	}

	return DDENUMRET_OK;
}

void TidyModes(void)
{
	RestoreMode(GetSMShwnd());

	if (pDirect3D)
	{
		IDirect3D7_Release(pDirect3D);
		pDirect3D = NULL;
	}

	if (pDirectDraw) 
	{
		IDirectDraw7_Release(pDirectDraw);
		pDirectDraw = NULL;
	}

	if (hModule)
	{
		FreeLibrary(hModule);
		hModule = NULL;
	}
}

BOOL ChangeMode(int ModeIndex, BOOL FullScreen)
{
	HWND			hwnd = GetSMShwnd();
	DDSURFACEDESC2	Surface;
	DDSCAPS2		SurfaceCaps;

	RestoreMode(hwnd);

	memset(&Surface, 0, sizeof(DDSURFACEDESC2));
	Surface.dwSize = sizeof(DDSURFACEDESC2);

	if (FullScreen)
	{
		if (IDirectDraw7_SetCooperativeLevel(pDirectDraw, hwnd, DDSCL_FULLSCREEN | DDSCL_EXCLUSIVE) != DD_OK) return FALSE;

		if (IDirectDraw7_SetDisplayMode(pDirectDraw, Mode[ModeIndex].Width, Mode[ModeIndex].Height, Mode[ModeIndex].BitDepth, 0, 0) != DD_OK)
		{
			RestoreMode(hwnd);

			return FALSE;
		}
	
		Surface.dwFlags			  = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
		Surface.ddsCaps.dwCaps	  = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX | DDSCAPS_3DDEVICE;
		Surface.dwBackBufferCount = 1;

		if (IDirectDraw7_CreateSurface(pDirectDraw, &Surface, &pFrontSurface, NULL) != DD_OK)
		{
			RestoreMode(hwnd);

			return FALSE;
		}

		memset(&SurfaceCaps, 0, sizeof(DDSCAPS2));
        SurfaceCaps.dwCaps = DDSCAPS_BACKBUFFER;

		if (IDirectDrawSurface7_GetAttachedSurface(pFrontSurface, &SurfaceCaps, &pBackSurface) != DD_OK)
		{
			RestoreMode(hwnd);

			return FALSE;
		}

		while (ShowCursor(FALSE) >= 0);
	}
	else
	{
		RECT					Rect;
		int						Width;
		int						Height;
		
		if (IDirectDraw7_SetCooperativeLevel(pDirectDraw, hwnd, DDSCL_NORMAL) != DD_OK) return FALSE;

        Surface.dwFlags			  = DDSD_CAPS;
        Surface.ddsCaps.dwCaps	  = DDSCAPS_PRIMARYSURFACE;

		if (IDirectDraw7_CreateSurface(pDirectDraw, &Surface, &pFrontSurface, NULL) != DD_OK)
		{
			return FALSE;
		}

		if (IDirectDraw7_CreateClipper(pDirectDraw, 0, &pClipper, NULL) != DD_OK)
		{
			RestoreMode(hwnd);

			return FALSE;
		}

		IDirectDrawClipper_SetHWnd(pClipper, 0, hwnd);
		IDirectDrawSurface7_SetClipper(pFrontSurface, pClipper);

		GetConsoleScreenRect(&Rect);
		Width  = (Rect.right - Rect.left) * RegistryInfo.ScreenScale;
		Height = (Rect.bottom - Rect.top) * RegistryInfo.ScreenScale;

		Surface.dwFlags			= DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
		Surface.dwWidth			= Width;
		Surface.dwHeight		= Height;
		Surface.ddsCaps.dwCaps	= DDSCAPS_OFFSCREENPLAIN | DDSCAPS_3DDEVICE;

		if (IDirectDraw7_CreateSurface(pDirectDraw, &Surface, &pBackSurface, NULL) != DD_OK)
		{
			RestoreMode(hwnd);

			return FALSE;
		}
	}

	ModeIsFullScreen = FullScreen;
	CurrentMode		 = ModeIndex;
	RegistryInfo.FullScreenMode = CurrentMode;

	if (!CreateTexture())
	{
		RestoreMode(hwnd);
		
		return FALSE;
	}

	if (IDirect3D7_CreateDevice(pDirect3D, &IID_IDirect3DHALDevice, pBackSurface, &p3DDevice) != D3D_OK)
	{
		RestoreMode(hwnd);

		return FALSE;
	}

	UpdateModeWindowCoords(hwnd);

	SetModeSmoothing(RegistryInfo.ScreenSmoothing);
	
	return TRUE;
}

void RestoreMode(HWND hwnd)
{
	while (ShowCursor(TRUE) < 0);

	DestroyTexture();
	
	if (p3DDevice)
	{
		IDirect3DDevice7_Release(p3DDevice);
		p3DDevice = NULL;
	}

	if (pBackSurface) 
	{
		IDirectDrawSurface7_Release(pBackSurface);
		pBackSurface = NULL;
	}

	if (pFrontSurface)
	{
		if (pClipper)
		{
			IDirectDrawSurface7_SetClipper(pFrontSurface, NULL);
			IDirectDrawClipper_Release(pClipper);
			pClipper = NULL;
		}

		IDirectDrawSurface7_Release(pFrontSurface);
		pFrontSurface = NULL;
	}

	if (ModeIsFullScreen)
	{
		IDirectDraw7_RestoreDisplayMode(pDirectDraw);
		IDirectDraw7_SetCooperativeLevel(pDirectDraw, GetSMShwnd(), DDSCL_NORMAL);
	}

	ModeIsFullScreen = FALSE;
}

void FlipScreens(void)
{
	if (UseGDISurface && ModeIsFullScreen) 
	{
		IDirectDrawSurface7_Blt(pFrontSurface, NULL, pBackSurface, NULL, DDBLT_WAIT, NULL);
	
		return;
	}
	
	while (1)
	{
		HRESULT		Result;

		if (ModeIsFullScreen)
		{
			Result = IDirectDrawSurface7_Flip(pFrontSurface, NULL, 0);
		}
		else
		{
			Result = IDirectDrawSurface7_Blt(pFrontSurface, &ScreenRect, pBackSurface, &ViewportRect, DDBLTFAST_WAIT, NULL);
		}

		if (Result == DD_OK) break;

		if (Result == DDERR_SURFACELOST)
		{
			Result = IDirectDrawSurface7_Restore(pFrontSurface);

			if (Result != DD_OK) break;
		}

		if (Result != DDERR_WASSTILLDRAWING) break;
	}
}

void UpdateModeWindowCoords(HWND hwnd)
{
	RECT			Rect;
	float			Top;
	float			Left;
	float			Right;
	float			Bottom;

	GetConsoleScreenRect(&Rect);

	if (ModeIsFullScreen)
	{
		float		Scale;
		float		ScaleX;
		float		ScaleY;
		int			Offset;
		
		ScaleX = (float)Mode[CurrentMode].Width / (float)(Rect.right - Rect.left);
		ScaleY = (float)Mode[CurrentMode].Height / (float)(Rect.bottom - Rect.top);

		if (ScaleX < ScaleY)
		{
			Scale = ScaleX;
		}
		else
		{
			Scale = ScaleY;
		}

		ViewportRect.right  = (int)((float)(Rect.right - Rect.left) * Scale);
		ViewportRect.bottom = (int)((float)(Rect.bottom - Rect.top) * Scale);

		Offset = (Mode[CurrentMode].Width - ViewportRect.right) / 2;

		ViewportRect.left   = Offset;
		ViewportRect.right += Offset;

		ClearBorders();
	}
	else
	{
		ViewportRect.top	= 0;
		ViewportRect.left	= 0;
		ViewportRect.right  = (Rect.right - Rect.left) * RegistryInfo.ScreenScale;
		ViewportRect.bottom = (Rect.bottom - Rect.top) * RegistryInfo.ScreenScale;

		GetClientRect(hwnd, &ScreenRect);
		ClientToScreen(hwnd, (LPPOINT)&ScreenRect.left);
		ScreenRect.right  = ScreenRect.left + ViewportRect.right;
		ScreenRect.bottom = ScreenRect.top + ViewportRect.bottom;
	}

	Top    = (float)Rect.top    / 256.0f;
	Left   = (float)Rect.left   / 256.0f;
	Right  = (float)Rect.right  / 256.0f;
	Bottom = (float)Rect.bottom / 256.0f;

	if (RegistryInfo.ScreenSmoothing)
	{
		Top	   += 1.0f / 512.0f;
		Left   += 1.0f / 512.0f;
		Right  -= 1.0f / 512.0f;
		Bottom -= 1.0f / 512.0f;
	}
	Vertex[0].sx = (float)ViewportRect.left;
	Vertex[0].sy = (float)ViewportRect.top;
	Vertex[0].tu = Left;
	Vertex[0].tv = Bottom;

	Vertex[1].sx = (float)ViewportRect.right;
	Vertex[1].sy = (float)ViewportRect.top;
	Vertex[1].tu = Right;
	Vertex[1].tv = Bottom;

	Vertex[2].sx = (float)ViewportRect.right;
	Vertex[2].sy = (float)ViewportRect.bottom;
	Vertex[2].tu = Right;
	Vertex[2].tv = Top;

	Vertex[3].sx = (float)ViewportRect.left;
	Vertex[3].sy = (float)ViewportRect.bottom;
	Vertex[3].tu = Left;
	Vertex[3].tv = Top;
}

BOOL CreateTexture(void)
{
	DDSURFACEDESC2			SurfaceDesc;

	IDirectDraw7_CreatePalette(pDirectDraw, DDPCAPS_8BIT | DDPCAPS_ALLOW256, (PALETTEENTRY *)GetSMSPalette(), &pPalette, NULL);

	memset(&SurfaceDesc, 0, sizeof(DDSURFACEDESC2));
	SurfaceDesc.dwSize			= sizeof(DDSURFACEDESC2);
	SurfaceDesc.dwFlags			= DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT | DDSD_LPSURFACE | DDSD_PITCH;
	SurfaceDesc.ddsCaps.dwCaps	= DDSCAPS_TEXTURE | DDSCAPS_SYSTEMMEMORY;
	SurfaceDesc.dwWidth			= 256;
	SurfaceDesc.dwHeight		= 256;
	SurfaceDesc.lpSurface		= (void*)bitmap.data;
	SurfaceDesc.lPitch			= 256;

	SurfaceDesc.ddpfPixelFormat.dwSize		  = sizeof(DDPIXELFORMAT);
	SurfaceDesc.ddpfPixelFormat.dwFlags		  = DDPF_PALETTEINDEXED8 | DDPF_RGB;
	SurfaceDesc.ddpfPixelFormat.dwRGBBitCount = 8;

	if (IDirectDraw7_CreateSurface(pDirectDraw, &SurfaceDesc, &pSystemTexture, NULL) != DD_OK)
	{
		DestroyTexture();

		return FALSE;
	}

    SurfaceDesc.dwFlags        = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT;
    SurfaceDesc.ddsCaps.dwCaps = DDSCAPS_TEXTURE | DDSCAPS_ALLOCONLOAD | DDSCAPS_VIDEOMEMORY;
	SurfaceDesc.lpSurface	   = 0;

	if (IDirectDraw7_CreateSurface(pDirectDraw, &SurfaceDesc, &pVRAMTexture, NULL) != DD_OK)
	{
		IDirectDrawSurface7_Release(pSystemTexture);

		SurfaceDesc.dwFlags						  = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT | DDSD_LPSURFACE | DDSD_PITCH;
		SurfaceDesc.ddsCaps.dwCaps				  = DDSCAPS_TEXTURE | DDSCAPS_SYSTEMMEMORY;
		SurfaceDesc.ddpfPixelFormat.dwSize		  = sizeof(DDPIXELFORMAT);
		SurfaceDesc.ddpfPixelFormat.dwFlags		  = DDPF_RGB;
		SurfaceDesc.ddpfPixelFormat.dwRGBBitCount = 16;
		SurfaceDesc.ddpfPixelFormat.dwRBitMask	  = 0x7C00;
		SurfaceDesc.ddpfPixelFormat.dwGBitMask	  = 0x03E0;
		SurfaceDesc.ddpfPixelFormat.dwBBitMask	  = 0x001F;
		SurfaceDesc.lPitch						  = 512;
		SurfaceDesc.lpSurface					  = (void*)bitmap.data;

		if (IDirectDraw7_CreateSurface(pDirectDraw, &SurfaceDesc, &pSystemTexture, NULL) != DD_OK)
		{
			DestroyTexture();

			return FALSE;
		}

	    SurfaceDesc.dwFlags        = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT;
		SurfaceDesc.ddsCaps.dwCaps = DDSCAPS_TEXTURE | DDSCAPS_ALLOCONLOAD | DDSCAPS_VIDEOMEMORY;
		SurfaceDesc.lpSurface	   = 0;
	
		if (IDirectDraw7_CreateSurface(pDirectDraw, &SurfaceDesc, &pVRAMTexture, NULL) != DD_OK)
		{
			DestroyTexture();

			return FALSE;
		}

		SwitchInternalRenderingFormat(16);
	}

	if (SurfaceDesc.ddpfPixelFormat.dwRGBBitCount == 8)
	{
		if (IDirectDrawSurface7_SetPalette(pSystemTexture, pPalette) != DD_OK)
		{
			DestroyTexture();

			return FALSE;
		}

		if (IDirectDrawSurface7_SetPalette(pVRAMTexture, pPalette) != DD_OK)
		{
			DestroyTexture();

			return FALSE;
		}
	}

	return TRUE;
}

void DestroyTexture(void)
{
	if (pVRAMTexture)
	{
		IDirectDrawSurface7_Release(pVRAMTexture);
		pVRAMTexture = NULL;
	}

	if (pSystemTexture)
	{
		IDirectDrawSurface7_Release(pSystemTexture);
		pSystemTexture = NULL;
	}

	if (pPalette)
	{
		IDirectDrawPalette_Release(pPalette);
		pPalette = NULL;
	}
}

BOOL UpdateTexture(void)
{
	RECT		Rect;

	GetConsoleScreenRect(&Rect);

	IDirectDrawPalette_SetEntries(pPalette, 0, 0, 256, (PALETTEENTRY *)GetSMSPalette());
	IDirectDrawSurface7_SetPalette(pSystemTexture, pPalette);
	IDirectDrawSurface7_SetPalette(pVRAMTexture, pPalette);

	if (IDirect3DDevice7_Load(p3DDevice, pVRAMTexture, (LPPOINT)&Rect.left, pSystemTexture, &Rect, 0) != D3D_OK)
	{
		return FALSE;
	}

	return TRUE;
}

BOOL DrawTexture(void)
{
	int			i;

	IDirect3DDevice7_SetTexture(p3DDevice, 0, pVRAMTexture);

	if (IDirect3DDevice7_BeginScene(p3DDevice) == DD_OK)
	{
		IDirect3DDevice7_DrawPrimitive(p3DDevice, D3DPT_TRIANGLEFAN, D3DFVF_TLVERTEX, Vertex, 4, 0);

		if (RegistryInfo.EnableScanLines)
		{
			for (i = 0 ; i < ViewportRect.bottom ; i += 2)
			{
				ScanLine[0].dvSX = 0;
				ScanLine[0].dvSY = (float)i;
				ScanLine[1].dvSX = (float)(Mode[CurrentMode].Width - 1);
				ScanLine[1].dvSY = (float)i;

				IDirect3DDevice7_DrawPrimitive(p3DDevice, D3DPT_LINELIST, D3DFVF_TLVERTEX, ScanLine, 2, 0);
			}
		}
	
		IDirect3DDevice7_EndScene(p3DDevice);
	}

	return TRUE;
}

void SetModeSmoothing(BOOL UseSmoothing)
{
	if (UseSmoothing)
	{
		IDirect3DDevice7_SetTextureStageState(p3DDevice, 0, D3DTSS_MINFILTER, D3DTFN_LINEAR);
		IDirect3DDevice7_SetTextureStageState(p3DDevice, 0, D3DTSS_MAGFILTER, D3DTFN_LINEAR);
	}
	else
	{
		IDirect3DDevice7_SetTextureStageState(p3DDevice, 0, D3DTSS_MINFILTER, D3DTFN_POINT);
		IDirect3DDevice7_SetTextureStageState(p3DDevice, 0, D3DTSS_MAGFILTER, D3DTFN_POINT);
	}
}

BOOL FullScreenMode(void)
{
	return ModeIsFullScreen;
}

void ClearBorders(void)
{
	DDBLTFX		BltFX;
	RECT		Rect;

	memset(&BltFX, 0, sizeof(DDBLTFX));
	BltFX.dwSize = sizeof(DDBLTFX);

	Rect.left   = 0;
	Rect.top    = 0;
	Rect.right  = ViewportRect.left;
	Rect.bottom = ViewportRect.bottom;

	IDirectDrawSurface7_Blt(pBackSurface, &Rect, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &BltFX);
	IDirectDrawSurface7_Blt(pFrontSurface, &Rect, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &BltFX);

	Rect.left   = ViewportRect.right;
	Rect.top    = 0;
	Rect.right  = Mode[CurrentMode].Width;
	Rect.bottom = ViewportRect.bottom;

	IDirectDrawSurface7_Blt(pBackSurface, &Rect, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &BltFX);
	IDirectDrawSurface7_Blt(pFrontSurface, &Rect, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &BltFX);
}

void FlipToGDISurface(BOOL UseGDI)
{
	if (ModeIsFullScreen)
	{
		if (UseGDI && !UseGDISurface)
		{
			UseGDISurface = TRUE;
			while (ShowCursor(TRUE) < 0);

			IDirectDraw7_CreateClipper(pDirectDraw, 0, &pClipper, NULL);
			IDirectDrawClipper_SetHWnd(pClipper, 0, GetSMShwnd());
			IDirectDraw7_FlipToGDISurface(pDirectDraw);
			IDirectDrawSurface7_SetClipper(pFrontSurface, pClipper);
			RedrawWindow(GetSMShwnd(), NULL, NULL, RDW_ERASE | RDW_FRAME | RDW_INVALIDATE | RDW_ERASENOW | RDW_UPDATENOW);
			IDirectDrawSurface7_Blt(pFrontSurface, NULL, pBackSurface, NULL, DDBLT_WAIT, NULL);
		}
		else if (!UseGDI && UseGDISurface)
		{
			IDirectDrawSurface7_SetClipper(pFrontSurface, NULL);
			IDirectDrawClipper_Release(pClipper);
			pClipper = NULL;

			while (ShowCursor(FALSE) >= 0);
			UseGDISurface = FALSE;

			ClearBorders();
		}
	}
}

void PopulateModesCombo(HWND hCombo)
{
	int			i;

	SendMessage(hCombo, CB_RESETCONTENT, 0, 0);
	
	for (i = 0 ; i < NoOfModes ; i++)
	{
		char	ModeName[32];

		wsprintf(ModeName, "%dx%dx%d", Mode[i].Width, Mode[i].Height, Mode[i].BitDepth);

		SendMessage(hCombo, CB_ADDSTRING, 0, (LPARAM)ModeName);
	}

	SendMessage(hCombo, CB_SETCURSEL, RegistryInfo.FullScreenMode, 0);
}