//============================================================================
//
//   SSSS    tt          lll  lll       
//  SS  SS   tt           ll   ll        
//  SS     tttttt  eeee   ll   ll   aaaa 
//   SSSS    tt   ee  ee  ll   ll      aa
//      SS   tt   eeeeee  ll   ll   aaaaa  --  "An Atari 2600 VCS Emulator"
//  SS  SS   tt   ee      ll   ll  aa  aa
//   SSSS     ttt  eeeee llll llll  aaaaa
//
// Copyright (c) 1995-1998 by Bradford W. Mott
//
// See the file "license" for information on usage and redistribution of
// this file, and for a DISCLAIMER OF ALL WARRANTIES.
//
// $Id: Props.hxx,v 1.2 2002/01/08 17:11:32 stephena Exp $
//============================================================================

#ifndef PROPERTIES_HXX
#define PROPERTIES_HXX

#include "bspf.hxx"

#ifdef __cplusplus
extern "C" {
#endif
#include "gpstdio.h"
#ifdef __cplusplus
}
#endif

/**
  This class represents objects which maintain a collection of 
  properties.  A property is a key and its corresponding value.

  A properties object can contain a reference to another properties
  object as its "defaults"; this second properties object is searched 
  if the property key is not found in the original property list.

  @author  Bradford W. Mott
  @version $Id: Props.hxx,v 1.2 2002/01/08 17:11:32 stephena Exp $
*/
class Properties
{
  public:
    /**
      Creates an empty properties object with the specified defaults.  The 
      new properties object does not claim ownership of the defaults.

      @param defaults The defaults
    */
    Properties(Properties* defaults = 0);

    /**
      Creates a properties list by copying another one

      @param properties The properties to copy
    */
    Properties(Properties& properties);

    /**
      Destructor
    */
    virtual ~Properties();

  public:
    /**
      Get the value assigned to the specified key.  If the key does
      not exist then the empty string is returned.

      @param key The key of the property to lookup
      @return The value of the property 
    */
    char* get(const char* key) const ;

    /**
      Set the value associated with key to the given value.

      @param key The key of the property to set
      @param value The value to assign to the property
    */
    void set(const char* key, const char* value);

  public:
    /**
      Load properties from the specified input stream

      @param in The input stream to use
    */
    bool load( unsigned char *md5loc );
	bool readpropline( unsigned char *loc, unsigned char *buf) ;
 
    /**
      Save properties to the specified output stream

      @param out The output stream to use
    */
    void save( F_HANDLE out );

  public:
    /**
      Read the next quoted string from the specified input stream
      and returns it.

      @param in The input stream to use
      @return The string inside the quotes
    */ 
    static void readQuotedString( F_HANDLE in, char* szBuf );
     
    /**
      Write the specified string to the given output stream as a 
      quoted string.

      @param out The output stream to use
      @param s The string to output
    */ 
    static void writeQuotedString( F_HANDLE out, const char* s);

  public:
    /**
      Overloaded assignment operator

      @param properties The properties object to set myself equal to
      @return Myself after assignment has taken place
    */
    Properties& operator = (Properties& properties);

  public:
    /**
      Helper function to perform a deep copy of the specified
      properties.  Assumes that old properties have already been 
      freed.

      @param properties The properties object to copy myself from
    */
    void copy(Properties& properties);

  private:
    // Structure used for storing properties
    struct Property 
    {
      char key[100];
      char value[100];
    };

    // Pointer to properties object to use for defaults or the null pointer
    //Properties* myDefaults;

    // Pointer to a dynamically allocated array of properties
    Property myProperties[25];

    // Current capacity of the properties array
    unsigned int myCapacity;

	public :
    // Size of the properties array (i.e. the number of <key,value> pairs)
    unsigned int mySize;
};
#endif
