//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

//---------------------------------------------------------------------------
*/

#include "neopop.h"
#include <windows.h>
#include <mmsystem.h>
#include <dsound.h>

#include "system_main.h"
#include "system_sound.h"
#include "system_config.h"

#include "sound.h"

//=============================================================================

static LPDIRECTSOUND8 ds = NULL;					// DirectSound Object
static LPDIRECTSOUNDBUFFER primaryBuffer = NULL;	// Primary Buffer
static LPDIRECTSOUNDBUFFER secondaryBuffer = NULL;	// Secondary Buffer

//Set by the init function, in samples: the data produced in a 59.95hz frame
DWORD blockLengthSamples;
_s16* block;	//Sound Chip output buffer, length = blockLengthSamples

//How long is the buffer?
DWORD bufferLengthBytes;
#define MIX_BUFFER	(0.2f)		//How much to buffer

//Write Cursor
DWORD LastWrite = 0xFFFFFFFF;

//=============================================================================

void system_sound_update(void)
{
	DWORD	pdwAudioBytes1, pdwAudioBytes2, count, status;
	_s16	*ppvAudioPtr1, *ppvAudioPtr2, *src, *dest;

	// Fill the sound buffer
	sound_update(block, blockLengthSamples);
	
	if (LastWrite == 0xFFFFFFFF)
		IDirectSoundBuffer_GetCurrentPosition(secondaryBuffer, NULL, &LastWrite);

	//Copy to Direct Sound
	if SUCCEEDED(IDirectSoundBuffer_Lock(secondaryBuffer, 
		LastWrite, blockLengthSamples << 1,
		&ppvAudioPtr1, &pdwAudioBytes1, &ppvAudioPtr2, &pdwAudioBytes2, 0))
	{
		src = block;	//Copy from this buffer

		dest = ppvAudioPtr1;
		count = pdwAudioBytes1 >> 1;
		while(count) { *dest++ = *src++; count--; }

		//Buffer Wrap?
		if (ppvAudioPtr2)
		{
		   dest = ppvAudioPtr2;
		   count = pdwAudioBytes2 >> 1;
		   while(count) { *dest++ = *src++; count--; }
		}
		
		IDirectSoundBuffer_Unlock(secondaryBuffer, 
			ppvAudioPtr1, pdwAudioBytes1, ppvAudioPtr2, pdwAudioBytes2);

		LastWrite += (blockLengthSamples << 1);
		if(LastWrite >= bufferLengthBytes) LastWrite -= bufferLengthBytes;
	}
	else
	{
		LastWrite = 0xFFFFFFFF;
		IDirectSoundBuffer_GetStatus(secondaryBuffer, &status);
		if (status & DSBSTATUS_BUFFERLOST)
		{
			if (IDirectSoundBuffer_Restore(secondaryBuffer) != DS_OK) return;
			IDirectSoundBuffer_Play(secondaryBuffer, 0, 0, DSBPLAY_LOOPING);
		}
	}
}

//=============================================================================

bool system_sound_init(void)
{
	WAVEFORMATEX wfx;
	DSBUFFERDESC dsbdesc;

    if FAILED(DirectSoundCreate8(NULL, &ds, NULL))
		return false;
 
    // Set co-op level
	if FAILED(IDirectSound_SetCooperativeLevel(ds, g_hWnd, DSSCL_PRIORITY ))
		if FAILED(IDirectSound_SetCooperativeLevel(ds, g_hWnd, DSSCL_NORMAL ))
				return false;

	//Create a primary buffer
	//=======================

	// Set up DSBUFFERDESC structure.
	ZeroMemory(&dsbdesc, sizeof(DSBUFFERDESC));
	dsbdesc.dwSize = sizeof(DSBUFFERDESC);
	dsbdesc.dwFlags = DSBCAPS_PRIMARYBUFFER;
	dsbdesc.dwBufferBytes = 0;
	dsbdesc.lpwfxFormat = NULL;

	if FAILED(IDirectSound_CreateSoundBuffer(ds, &dsbdesc, &primaryBuffer, NULL))
		return false;
 
    // Set buffer format
    memset(&wfx, 0, sizeof(WAVEFORMATEX)); 
    wfx.wFormatTag = WAVE_FORMAT_PCM; 
    if (stereo)	wfx.nChannels = 2; else wfx.nChannels = 1;
    wfx.nSamplesPerSec = sound_frequency;	
    wfx.wBitsPerSample = 16;
    wfx.nBlockAlign = (wfx.wBitsPerSample / 8) * wfx.nChannels;
    wfx.nAvgBytesPerSec = wfx.nSamplesPerSec * wfx.nBlockAlign;

 	if FAILED(IDirectSoundBuffer_SetFormat(primaryBuffer, &wfx))
		return false;

	if FAILED(IDirectSoundBuffer_Play(primaryBuffer, 0,0, DSBPLAY_LOOPING ))
		return false;

	//Calculate the length of a frame in samples
	// 0.01668 is the number of seconds per frame
	blockLengthSamples = (DWORD)((double)sound_frequency * 0.01668);
	if (stereo)	blockLengthSamples <<= 1;

	//Allocate a buffer to fill with sound samples.
	block = (_s16*)malloc(blockLengthSamples * sizeof(_s16));

	//Calculate good buffer size (*2 for 16bit)
	bufferLengthBytes = (DWORD)(MIX_BUFFER * sound_frequency * 2);
	if (stereo)	bufferLengthBytes <<= 1;

	//Create a secondary buffer
	//=========================
	
	// Set up DSBUFFERDESC structure.
	memset(&dsbdesc, 0, sizeof(DSBUFFERDESC));
	dsbdesc.dwSize = sizeof(DSBUFFERDESC);
	dsbdesc.dwFlags = DSBCAPS_LOCSOFTWARE | DSBCAPS_GETCURRENTPOSITION2;
	dsbdesc.lpwfxFormat = &wfx;
	dsbdesc.dwBufferBytes = bufferLengthBytes;

	// Try to create a secondary buffer.
	if FAILED(IDirectSound_CreateSoundBuffer(ds, &dsbdesc, &secondaryBuffer, NULL))
		return false;

	//Initialises sound chips, matching frequncies
	sound_init(sound_frequency);

	LastWrite=0xffffffff;

	return true;
}

//=============================================================================

void system_sound_shutdown(void)
{
	if (secondaryBuffer)	IDirectSoundBuffer_Stop(secondaryBuffer);
	if (secondaryBuffer)	IDirectSoundBuffer_Release(secondaryBuffer);
	if (primaryBuffer)		IDirectSoundBuffer_Stop(primaryBuffer);
	if (primaryBuffer)		IDirectSoundBuffer_Release(primaryBuffer);
	if (ds)	IDirectSound_Release(ds);

	LastWrite=0xffffffff;
	free(block);
}

//=============================================================================

void system_sound_start(void)
{
	BYTE	*ppvAudioPtr1, *ppvAudioPtr2;
	DWORD	pdwAudioBytes1, pdwAudioBytes2;

	if (secondaryBuffer)
	{
		IDirectSoundBuffer_Stop(secondaryBuffer);

		// Fill the sound buffer
		if SUCCEEDED(IDirectSoundBuffer_Lock(secondaryBuffer, 0, 0, 
			&ppvAudioPtr1, &pdwAudioBytes1, 
			&ppvAudioPtr2, &pdwAudioBytes2, DSBLOCK_ENTIREBUFFER))
		{
			if (ppvAudioPtr1 && pdwAudioBytes1)
				memset(ppvAudioPtr1, 0, pdwAudioBytes1);

			if (ppvAudioPtr2 && pdwAudioBytes2)
				memset(ppvAudioPtr2, 0, pdwAudioBytes2);
			
			IDirectSoundBuffer_Unlock(secondaryBuffer, 
				ppvAudioPtr1, pdwAudioBytes1, ppvAudioPtr2, pdwAudioBytes2);
		}

		if (mute == false)
			IDirectSoundBuffer_Play(secondaryBuffer, 0,0, DSBPLAY_LOOPING );
	}

	//Initialises sound chips, matching frequncies
	sound_init(sound_frequency);

	LastWrite=0xffffffff;
}

void system_sound_stop(void)
{
	BYTE	*ppvAudioPtr1, *ppvAudioPtr2;
	DWORD	pdwAudioBytes1, pdwAudioBytes2;

	if (secondaryBuffer)
	{
		IDirectSoundBuffer_Stop(secondaryBuffer);

		// Stop any residual noise when switching to a rom without sound.
		if SUCCEEDED(IDirectSoundBuffer_Lock(secondaryBuffer, 0, 0, 
			&ppvAudioPtr1, &pdwAudioBytes1, 
			&ppvAudioPtr2, &pdwAudioBytes2, DSBLOCK_ENTIREBUFFER))
		{
			if (ppvAudioPtr1 && pdwAudioBytes1)
				memset(ppvAudioPtr1, 0, pdwAudioBytes1);

			if (ppvAudioPtr2 && pdwAudioBytes2)
				memset(ppvAudioPtr2, 0, pdwAudioBytes2);
			
			IDirectSoundBuffer_Unlock(secondaryBuffer, 
				ppvAudioPtr1, pdwAudioBytes1, ppvAudioPtr2, pdwAudioBytes2);
		}
	}

	LastWrite=0xffffffff;
}

//=============================================================================
