//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

  21 JUL 2002 - neopop_uk
=======================================
- Moved the button settings to the 'system_config.h' file

//---------------------------------------------------------------------------
*/

#include <windows.h>
#define DIRECTINPUT_VERSION 0x0800
#include <dinput.h>

extern "C"
{
#include "neopop.h"
#include "system_main.h"
#include "system_config.h"
#include "mem.h"
}

//=============================================================================

static LPDIRECTINPUT8	di = NULL;		// DirectInput Device
static LPDIRECTINPUTDEVICE8 joy = NULL; // DirectInput Joystick
static LPDIRECTINPUTDEVICE8 keyboard = NULL; // DirectInput Keyboard
static DIDEVCAPS caps;

#define INPUT_MAXAXISVALUE 64
#define DEADZONE	16

//This value toggles on and off to create the turbo effect.
static bool auto_flipflop = false;

//I/O state
static _u8 up,down,left,right,button_a,button_b,option;

//=============================================================================

//-----------------------------------------------------------------------------
// setDIRange()      
//-----------------------------------------------------------------------------
static HRESULT setDIRange(DWORD obj, long max, long min)
{
	DIPROPRANGE diprg;

	diprg.diph.dwSize       = sizeof(diprg);
	diprg.diph.dwHeaderSize = sizeof(diprg.diph);
	diprg.diph.dwObj        = obj;
	diprg.diph.dwHow        = DIPH_BYOFFSET;
	diprg.lMin              = min; 
	diprg.lMax              = max; 

	return joy->SetProperty(DIPROP_RANGE, &diprg.diph);
}

//-----------------------------------------------------------------------------
// EnumJoysticksCallback()
//-----------------------------------------------------------------------------
BOOL CALLBACK EnumJoysticksCallback(const DIDEVICEINSTANCE* pdidInstance, void* data)
{
    HRESULT hr;

    // Obtain an interface to the enumerated joystick.
    if FAILED(hr = di->CreateDevice(pdidInstance->guidInstance, &joy, NULL))
        return DIENUM_CONTINUE;

	if FAILED(hr = joy->SetDataFormat(&c_dfDIJoystick2))
	{
		joy->Release();	joy = NULL;
		return DIENUM_STOP;
	}

	if FAILED(hr = joy->SetCooperativeLevel(g_hWnd, DISCL_EXCLUSIVE | DISCL_FOREGROUND))
	{
		joy->Release();	joy = NULL;
		return DIENUM_STOP;
	}

	//X-AXIS
	if FAILED(hr = setDIRange(DIJOFS_X, +INPUT_MAXAXISVALUE, -INPUT_MAXAXISVALUE))
	{
		joy->Release();	joy = NULL;
		return DIENUM_STOP;
	}
		
	//Y-AXIS
	if FAILED(hr = setDIRange(DIJOFS_Y, +INPUT_MAXAXISVALUE, -INPUT_MAXAXISVALUE))
	{
		joy->Release();	joy = NULL;
		return DIENUM_STOP;
	}

	joy->Acquire();	// Try to Acquire the joystick

	// Get the Caps
	ZeroMemory(&caps, sizeof(DIDEVCAPS));
	caps.dwSize = sizeof(DIDEVCAPS);
	joy->GetCapabilities(&caps);

    return DIENUM_STOP;
}
 
//-----------------------------------------------------------------------------
// key_init()
//-----------------------------------------------------------------------------
static bool key_init(void)
{
	// Create a device
	if FAILED(di->CreateDevice(GUID_SysKeyboard, &keyboard, NULL))
		return false;

	// Set data format
	if FAILED(keyboard->SetDataFormat(&c_dfDIKeyboard))
	{
		keyboard->Release();	keyboard = NULL;
		return false;
	}

	// Set the cooperative level of the keyboard 
 	if FAILED(keyboard->SetCooperativeLevel(g_hWnd, 
			DISCL_FOREGROUND | DISCL_NONEXCLUSIVE))
	{
		keyboard->Release();	keyboard = NULL;
		return false;
	}

	keyboard->Acquire();	// Try to Acquire the keyboard
	return true;
}

//-----------------------------------------------------------------------------
// input_init()
//-----------------------------------------------------------------------------
extern "C" int system_input_init(void)
{
	// Create DirectInput8 interface
	if FAILED(DirectInput8Create(g_hInstance, DIRECTINPUT_VERSION, 
		IID_IDirectInput8,(void**)&di, NULL))
		return 0;

	// Find a joystick
	di->EnumDevices(DI8DEVCLASS_GAMECTRL, EnumJoysticksCallback,
		NULL, DIEDFL_ATTACHEDONLY);

	key_init();

	if (joy || keyboard)
		return 1;
	else
		return 0;
}

//-----------------------------------------------------------------------------
// shutdown()
//-----------------------------------------------------------------------------
extern "C" void system_input_shutdown(void)
{
	if (joy != NULL)
	{
		joy->Unacquire();
		IDirectInputDevice8_Release(joy);
	}

	if (keyboard != NULL)
	{
		keyboard->Unacquire();
		IDirectInputDevice8_Release(keyboard);
	}

	if (di != NULL)		IDirectInput8_Release(di);
}

//-----------------------------------------------------------------------------
// buttonCount()
//-----------------------------------------------------------------------------
extern "C" int system_input_buttoncount(void)
{
	if (joy != NULL)
		return caps.dwButtons;
	else
		return -1;	//No stick
}

//-----------------------------------------------------------------------------
// buttonName()
//-----------------------------------------------------------------------------
char button_name[_MAX_PATH];
int count = 0;

BOOL CALLBACK DIEnumDeviceObjectsCallback(LPCDIDEVICEOBJECTINSTANCE obj,
										  void* ref) 
{
	int i = *(int*)ref;
	if (count == i)
	{
		sprintf(button_name, "%s", obj->tszName);
		return DIENUM_STOP;
	}

	count++;
	return DIENUM_CONTINUE;
} 

extern "C" char* system_input_buttonname(int i)
{
	count = 0;
	sprintf(button_name, "Button %d", i+1);
	joy->EnumObjects(DIEnumDeviceObjectsCallback, (void*)&i, DIDFT_BUTTON); 
	return strdup(button_name);
}

//-----------------------------------------------------------------------------
// system_input_update()
//-----------------------------------------------------------------------------
extern "C" void system_input_update(void)
{
	//Clear the controller status
	up = down = left = right = button_a = button_b = option = 0;

	//Autofire flip flop.
	auto_flipflop = !auto_flipflop;

    if (joy)	//Check to see if the joystick is active
	{
		DIJOYSTATE2 js;           // Direct Input joystick state 

		if FAILED(joy->Poll())
		{
			joy->Acquire();
			return;
		}

		if FAILED(joy->GetDeviceState(sizeof(DIJOYSTATE2), &js))
		{
			joy->Acquire();
			return;
		}

		//D-Pad
		if (js.lX < -DEADZONE)	left = 1;
		if (js.lX > +DEADZONE)	right = 1;
		if (js.lY < -DEADZONE)	up = 1;
		if (js.lY > +DEADZONE)	down = 1;

		//Buttons
		if (js.rgbButtons[BUTTONA] & 0x80)	button_a = 1;
		if (js.rgbButtons[BUTTONB] & 0x80)	button_b = 1;
		if (js.rgbButtons[BUTTONO] & 0x80)	option = 1;

		//Autofire
		if (AUTOA && auto_flipflop)		button_a = 0;
		if (AUTOB && auto_flipflop)		button_b = 0;
	}

	if (keyboard) 
    {
		char ks[256];
 
		if FAILED(keyboard->GetDeviceState(sizeof(ks), &ks))
		{
			keyboard->Acquire();
			return;
		}

		//D-Pad
		if (ks[KEYL] & 0x80)	left = 1;
		if (ks[KEYR] & 0x80)	right = 1;
		if (ks[KEYU] & 0x80)	up = 1;
		if (ks[KEYD] & 0x80)	down = 1;

		//Buttons
		if (ks[KEYA] & 0x80)	button_a = 1;
		if (ks[KEYB] & 0x80)	button_b = 1;
		if (ks[KEYO] & 0x80)	option = 1;
    }

	//Write the controller status to memory
	ram[0x6F82] = up | (down << 1) | (left << 2) | (right << 3) | 
		(button_a << 4) | (button_b << 5) | (option << 6);
}

//=============================================================================
