//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

21 JUL 2002 - neopop_uk
=======================================
- Tidied up and removed obsolete functions.
- Added declarations for the new IO functions.

//---------------------------------------------------------------------------
*/

#ifndef __NEOPOP__
#define __NEOPOP__
//=============================================================================

// #####################
// ### BUILD OPTIONS ###
// #####################

//Comment out to use the raw bios functions (more accurate? a little),
//otherwise the functions are emulated in a high-level way (faster!)
#define BIOSHLE	

//Comment out to disable the quick bios, and use the bios menu (more authentic)
//Not possible in HLE mode, and not yet supported in raw mode.
#define QUICKBIOS

//=============================================================================

// ### DO NOT ALTER THIS, IT IS ONLY FOR LOGICAL CONSISTENCY
//The QUICKBIOS option can only be disabled when #define BIOSHLE is disabled,
//otherwise the system is always started in quick mode.
#ifndef QUICKBIOS
#ifdef BIOSHLE
#define QUICKBIOS
#endif
#endif

//=============================================================================

#ifdef NEOPOP_DEBUG
#define PROGRAM_NAME "NeoPop (Debugger)"
#else
#define PROGRAM_NAME "NeoPop"
#endif

#define NEOPOP_VERSION	"Version 0.52 (" __DATE__ ")"

//=============================================================================

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

//TYPES
typedef unsigned char		_u8;
typedef unsigned short		_u16;
typedef unsigned long		_u32;
typedef unsigned __int64	_u64;

typedef	signed char			_s8;
typedef signed short		_s16;
typedef signed long			_s32;
typedef signed __int64		_s64;

#ifndef __cplusplus
typedef enum{false,true}	bool;
#endif

//=============================================================================

//-----------------------------------------------------------------------------
// Core Configuration
//-----------------------------------------------------------------------------

extern bool language_english;
extern bool system_colour;
extern bool stereo;
extern _u16 sound_frequency;
extern bool doneFrame ;

//=============================================================================

//-----------------------------------------------------------------------------
// Emulator <--> System Interface
//-----------------------------------------------------------------------------

/*! Used to generate a critical message for the user. After the message
	has been displayed, the function should return. The message is not
	necessarily a fatal error. */
	
	void __cdecl system_message(char* vaMessage,...);


/*! Called at the start of the vertical blanking period, this function is
	designed to perform many of the critical hardware interface updates
	Here is a list of recommended actions to take:
	
	- The frame buffer should be copied to the screen.
	- The frame rate should be throttled to 59.95hz
	- The sound chips should be polled for the next chunk of data
	- Input should be polled and the current status written to "ram[0x6F82]" */
	
	void system_VBL(void);


/*! Clears the display. Called at reset time. Recommended that the display
	is cleared to white, not black. */

	void system_graphics_clear(void);


/*! Starts sound output with an empty buffer, and a call to reset the sound
	chips using the current sound_frequency */
	
	void system_sound_start(void);


/*!	Emulate a single instruction with correct TLCS900h:Z80 timing */

	void emulate(void);


//-----------------------------------------------------------------------------
// Emulator <--> Debugger Interface
//-----------------------------------------------------------------------------

#ifdef NEOPOP_DEBUG

	//Debugger message filters
	//(TRUE = allow messages to be generated)
	extern bool filter_mem;
	extern bool filter_bios;
	extern bool filter_dma;
	extern bool filter_sound;


/*! Emulate a single instruction in Debug mode, checking for exceptions */

	void emulate_debug(bool dis_TLCS900h, bool dis_Z80);


/*!	Generates a debugger specific message that is not relevant to the
	main build. For example the memory module (mem.c) uses this function
	to indicate a memory exception */

	void __cdecl system_debug_message(char* vaMessage,...);


/*! This function pairs with the function above. This is used to associate
	an additional address to the last debug message. It was added so that
	later the message could be selected and an appropriate address retrieved.
	For example a memory exception message might have the address of the
	instruction that caused the error, this way the instruction could be
	displayed, or used as a breakpoint, etc. */

	void system_debug_message_associate_address(_u32 address);


/*!	Signals the debugger to stop execution of program code and await user
	instruction. This is called when resetting, or after a instruction or 
	memory error. */

	void system_debug_stop(void);


/*! Signals the debugger to refresh it's information as the system state
	has changed. For example after loading a saved state, a memory viewer
	will no longer be displaying current information. */

	void system_debug_refresh(void);
	

/*!	Indicates to the debugger that the instruction located at the current
	program counter value is about to be executed, and this address should
	be stored in the execution history. */

	void system_debug_history_add(void);
	

/*! Signals the debugger to clear it's list of instruction history. This
	is called by the reset function to correctly indicate that there have
	been no previous instructions executed. */

	void system_debug_history_clear(void);

#endif


//-----------------------------------------------------------------------------
// Emulator <--> IO Interface
//-----------------------------------------------------------------------------

/*! Reads as much of the file specified by 'filename' into the given, 
	preallocated buffer. This is rom data */

	bool system_io_rom_read(char* filename, _u8* buffer, _u32 bufferLength);

		//=========================================

/*! Reads from the file specified by 'filename' into the given preallocated
	buffer. This is flash data. The filename may need to be shortened. */

	bool system_io_flash_read(char* filename, _u8* buffer, _u32 bufferLength);

/*! Writes to the file specified by 'filename' from the given buffer.
	This is flash data. The filename may need to be shortened */
	
	bool system_io_flash_write(char* filename, _u8* buffer, _u32 bufferLength);

		//=========================================

/*! Reads from the file specified by 'filename' into the given preallocated
	buffer. This is state data. The filename may need to be shortened. */

	bool system_io_state_read(char* filename, _u8* buffer, _u32 bufferLength);
	
/*! Writes to the file specified by 'filename' from the given buffer.
	This is state data. The filename may need to be shortened */

	bool system_io_state_write(char* filename, _u8* buffer, _u32 bufferLength);


//=============================================================================
#endif
