//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

//---------------------------------------------------------------------------
*/

#include "neopop.h"
#include "TLCS900H_interpret.h"
#include "TLCS900H_registers.h"
#include "Z80_interface.h"
#include "interrupt.h"
#include "mem.h"
#include "disassemble.h"
#include "bios.h"

//=============================================================================

bool language_english = true;
bool system_colour = true;
bool doneFrame ;
//=============================================================================

static bool debug_abort_instruction = false;
void __cdecl instruction_error(char* vaMessage,...)
{
	char message[1000];
	va_list vl;

	va_start(vl, vaMessage);
	vsprintf(message, vaMessage, vl);
	va_end(vl);

#ifdef NEOPOP_DEBUG
	system_debug_message(message);
	debug_abort_instruction = true;
#else
	system_message("[PC %06X] %s", pc, message);
#endif
}

//=============================================================================

#ifndef NEOPOP_DEBUG

int emulate_clockratio_TLCS900h = 16;
int emulate_clockratio_Z80 = 8;

void emulate(void)
{
	_u8 i, TLCS900h_cycles = 0;
	
	//TLCS900h instructions
	for (i = 0; i < emulate_clockratio_TLCS900h; i++)
		TLCS900h_cycles += TLCS900h_interpret();

	//Z80 instructions
	if (Z80ACTIVE)
		for (i = 0; i < emulate_clockratio_Z80; i++)
			Z80EMULATE

	//Update timers after a few instructions
	updateTimers(TLCS900h_cycles);
}

#endif

//=============================================================================

#ifdef NEOPOP_DEBUG

//Debug Message Filters
bool filter_sound;		
bool filter_bios;		
bool filter_dma;		
bool filter_mem;		

void emulate_debug(bool dis_TLCS900h, bool dis_Z80)
{
	_u32 storePC = pc;

	debug_abort_memory = false;
	debug_abort_instruction = false;

	system_debug_history_add();		//For the debugger

	if (dis_TLCS900h)
	{
		char* s;

		//Disassemble TLCS-900h
		_u32 oldpc = pc;
		s = disassemble();
		system_debug_message(s);
		system_debug_message_associate_address(oldpc);
		free(s);
		pc = oldpc;
	}

	if (dis_Z80)
	{
		//Disassemble Z80
		if (Z80ACTIVE)
		{
			char* s;
			_u16 pc = Z80_getReg(Z80_REG_PC);
			_u16 store_pc = pc;

			//Disassemble
			s = Z80_disassemble(&pc);
			system_debug_message(s);
			system_debug_message_associate_address(store_pc + 0x7000);
			free(s);
		}
	}

	debug_abort_memory = false;
	debug_abort_instruction = false;

	//==================
	// EMULATE
	//==================
	{
		//TLCS900h instruction
		_u8 TLCS900h_cycles = TLCS900h_interpret();

		//Z80 Instruction
		if (Z80ACTIVE) Z80EMULATE;

		updateTimers(TLCS900h_cycles);
	}

	//Check register code error
	if (rErr != RERR_VALUE)
		instruction_error("Invalid register code used.");

	//Memory Exception
	if (debug_abort_memory && filter_mem)
	{
		_u32 oldpc = pc;
		char* s;

		debug_abort_memory = false;

		//Try to disassemble the erroneous instruction
		pc = storePC;
		debug_mask_memory_error_messages = true;
		s = disassemble();
		debug_mask_memory_error_messages = false;

		if (debug_abort_memory == false)
		{
			system_debug_message("Stopped due to memory exception caused by");
			system_debug_message("     %s", s);
			system_debug_message_associate_address(storePC);
			system_debug_message("\n");
		}
		else
		{
			system_debug_message("Stopped due to memory exception caused at %06X", storePC);
			system_debug_message_associate_address(storePC);
		}
		free(s);
		pc = oldpc;
		
		system_debug_stop();
		system_debug_refresh();
		return;
	}

	//Unimplemented Instruction
	if (debug_abort_instruction)
	{
		_u32 oldpc = pc;
		char* s;

		debug_abort_memory = false;

		//Try to disassemble the erroneous instruction
		pc = storePC;
		debug_mask_memory_error_messages = true;
		s = disassemble();
		debug_mask_memory_error_messages = false;

		if (debug_abort_memory == false)
		{
			system_debug_message("Stopped due to instruction");
			system_debug_message("     %s", s);
			system_debug_message_associate_address(storePC);
			system_debug_message("\n");
		}
		else
		{
			system_debug_message("Stopped due to instruction at %06X", storePC);
			system_debug_message_associate_address(storePC);
		}
		free(s);
		pc = oldpc;
		
		system_debug_stop();
		system_debug_refresh();
		return;
	}
}

#endif

//=============================================================================

