//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

//---------------------------------------------------------------------------
*/

#include "neopop.h"
#include "TLCS900H_registers.h"
#include "Z80_interface.h"
#include "rom.h"
#include "bios.h"
#include "gfx.h"
#include "mem.h"
#include "eeprom.h"
#include "interrupt.h"
#include "sound.h"

//=============================================================================

_u8 ram[1 + RAM_END - RAM_START];

bool debug_abort_memory;
bool debug_mask_memory_error_messages;

bool memory_unlock_flash_write = false;
bool memory_flash_error = false;

//=============================================================================

#ifdef NEOPOP_DEBUG
static void memory_error(_u32 address, bool read)
{
	debug_abort_memory = true;

	if (filter_mem)
	{
		if (debug_mask_memory_error_messages)
			return;

		if (read)
			system_debug_message("Memory Exception: Read from %06X", address);
		else
			system_debug_message("Memory Exception: Write to %06X", address);
	}
}
#endif

//=============================================================================

void* translate_address_read(_u32 address)
{
	address &= 0xFFFFFF;

	// ===================================

#ifdef NEOPOP_DEBUG
	// ===================================
	if (address == 0)	{ memory_error(address, true); return NULL; }

	//Print bios calls (0 - 15 (0xFFFE3C))
	if (filter_bios && address >= 0xFFFE00 && address <= 0xFFFE3C)
		biosDecode((address - 0xFFFE00) >> 2);
	//Print comms bios calls (16 - 26 (0xFFFE68))
//	if (filter_bios && address >= 0xFFFE40 && address <= 0xFFFE68)
//		biosDecode((address - 0xFFFE00) >> 2);

	// ===================================
#endif

	// ===================================

	if (address <= RAM_END)
		return ram + address;

	//ROM (LOW)
	if (rom.data && address >= ROM_START && address <= ROM_END)
	{
		if (address <= ROM_START + rom.length)
			return rom.data + (address - ROM_START);
		else
			return NULL;
	}

	//ROM (HIGH)
	if (rom.data && address >= HIROM_START && address <= HIROM_END)
	{
		if (address <= HIROM_START + (rom.length - 0x200000))
			return rom.data + 0x200000 + (address - HIROM_START);
		else
			return NULL;
	}

	//BIOS Access
	if ((address & 0xFF0000) == 0xFF0000)
		return bios + (address & 0xFFFF); // BIOS ROM

	//Detect a flash memory error
	if (memory_unlock_flash_write)
		memory_flash_error = true;

#ifdef NEOPOP_DEBUG
	memory_error(address, true);
#endif
	return NULL;
}

//=============================================================================

void* translate_address_write(_u32 address)
{	
	address &= 0xFFFFFF;

	// ===================================

#ifdef NEOPOP_DEBUG
	if (address == 0)
	{
		memory_error(address, false);
		return NULL;
	}
	if (address > 0 && address <= RAM_END)
		return ram + address;
#else
	// ===================================
	if (address <= RAM_END)
		return ram + address;
#endif

	if (memory_unlock_flash_write)
	{
		//ROM (LOW)
		if (rom.data && address >= ROM_START && address <= ROM_END)
		{
			if (address <= ROM_START + rom.length)
				return rom.data + (address - ROM_START);
			else
				return NULL;
		}

		//ROM (HIGH)
		if (rom.data && address >= HIROM_START && address <= HIROM_END)
		{
			if (address <= HIROM_START + (rom.length - 0x200000))
				return rom.data + 0x200000 + (address - HIROM_START);
			else
				return NULL;
		}

		memory_flash_error = true;
	}

#ifdef NEOPOP_DEBUG
	memory_error(address, false);
#endif
	return NULL;
}

//=============================================================================

void post_write(_u32 address)
{
	//Direct Sound Access
	if ((*(_u16*)(ram + 0xb8)) == 0xAA55)
	{
		if (address == 0xA1)	Write_SoundChipLeft(ram[0xA1]);
		if (address == 0xA0)	Write_SoundChipRight(ram[0xA0]);
	}

	//DAC Write
	if (address == 0xA2)
	{
		dac_write(ram[0xA2], ram[0xA3]);
	}

	//Clear counters?
	if (address == 0x20)
	{
		_u8 TRUN = ram[0x20];

		if ((TRUN & 0x01) == 0)		timer[0] = 0;
		if ((TRUN & 0x02) == 0)		timer[1] = 0;
		if ((TRUN & 0x04) == 0)		timer[2] = 0;
		if ((TRUN & 0x08) == 0)		timer[3] = 0;
	}

	//z80 - NMI
	if (address == 0xBA)
		Z80_nmi();
}

//=============================================================================

_u8 loadB(_u32 address)
{
	_u8* ptr = translate_address_read(address);
	if (ptr == NULL)
		return 0;
	else
		return *ptr;
}

_u16 loadW(_u32 address)
{
	_u16* ptr = translate_address_read(address);
	if (ptr == NULL)
		return 0;
	else
		return *ptr;
}

_u32 loadL(_u32 address)
{
	_u32* ptr = translate_address_read(address);
	if (ptr == NULL)
		return 0;
	else
		return *ptr;
}

//=============================================================================

void storeB(_u32 address, _u8 data)
{
	_u8* ptr = translate_address_write(address);

	//Write
	if (ptr)
	{
		*ptr = data;
		post_write(address);
	}
}

void storeW(_u32 address, _u16 data)
{
	_u16* ptr = translate_address_write(address);

	//Write
	if (ptr)
	{
		*ptr = data;
		post_write(address);
	}
}

void storeL(_u32 address, _u32 data)
{
	_u32* ptr = translate_address_write(address);

	//Write
	if (ptr)
	{
		*ptr = data;
		post_write(address);
	}
}

//=============================================================================

void reset_memory(void)
{
	//RAM
	memset(ram, 0, sizeof(ram));

	ram[0x006F] = 0x4E;	//Watchdog timer
	
	ram[0x00B8] = 0xAA;	//No z80 at boot
	ram[0x00B9] = 0xAA;
	
	ram[0x6F80] = 0xFF;	//Lots of battery power!
	ram[0x6F81] = 0x03;

	//Language: 0 = Japanese, 1 = English
	ram[0x6F87] = (_u8)language_english;	

	//Color Mode Selection: 0x00 = B&W, 0x10 = Colour
	ram[0x6F91] = 0x10;	//Colour bios

	if (system_colour)
	{
		ram[0x6F95] = 0x10;
	}
	else
	{
		ram[0x6F95] = 0x00;
	}

	ram[0x6F84] = 0x40;	// "Power On" startup
	ram[0x6F85] = 0x00;	// No shutdown request
	ram[0x6F86] = 0x00;	// No user answer (?)

	ram[0x8006] = 0xc6;	// Frame Rate Register

	ram[0x83F0] = 0xFF;	// Default border colour
	ram[0x83F1] = 0x0F;

	ram[0x8400] = 0xFF;	// LED on
	ram[0x8402] = 0x80;	// Flash cycle = 1.3s
	
	// ==========================

	//EEPROM
	eepromResetDecode();
}

//=============================================================================


