//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

//---------------------------------------------------------------------------
*/

#include "neopop.h"
#include "mem.h"
#include "gfx.h"

//=============================================================================

static void Plot(_u8 x, _u16 pal_address, _u8 palette, _u8 index, _u8 depth)
{
	//Clip
	if (index == 0 || x < winx || x >= (winw + winx) || x >= SCREEN_WIDTH)
		return;

	//Depth check, <= to stop later sprites overwriting pixels!
	if (depth <= zbuffer[x]) return;
	zbuffer[x] = depth;

	//Get the colour of the pixel
	data32 = *(_u16*)(ram + pal_address + (palette << 3) + (index << 1));

	r = (data32 & 0x000F) << 20;
	g = (data32 & 0x00F0) << 8;
	b = (data32 & 0x0F00) >> 4;

	cfb[x + cfb_offset] = r | g | b;
}

static void drawPattern(_u8 screenx, _u16 tile, _u8 tiley, bool mirror, 
				 _u16 pal_address, _u8 pal, _u8 depth)
{
	//Get the data for th e "tiley'th" line of "tile".
	_u16 data = *(_u16*)(ram + 0xA000 + (tile * 16) + (tiley * 2));

	//Horizontal Flip
	if (mirror)
	{
		Plot(screenx + 7, pal_address, pal, (data & 0xC000) >> 0xE, depth);
		Plot(screenx + 6, pal_address, pal, (data & 0x3000) >> 0xC, depth);
		Plot(screenx + 5, pal_address, pal, (data & 0x0C00) >> 0xA, depth);
		Plot(screenx + 4, pal_address, pal, (data & 0x0300) >> 0x8, depth);
		Plot(screenx + 3, pal_address, pal, (data & 0x00C0) >> 0x6, depth);
		Plot(screenx + 2, pal_address, pal, (data & 0x0030) >> 0x4, depth);
		Plot(screenx + 1, pal_address, pal, (data & 0x000C) >> 0x2, depth);
		Plot(screenx + 0, pal_address, pal, (data & 0x0003) >> 0x0, depth);
	}
	else
	//Normal
	{
		Plot(screenx + 0, pal_address, pal, (data & 0xC000) >> 0xE, depth);
		Plot(screenx + 1, pal_address, pal, (data & 0x3000) >> 0xC, depth);
		Plot(screenx + 2, pal_address, pal, (data & 0x0C00) >> 0xA, depth);
		Plot(screenx + 3, pal_address, pal, (data & 0x0300) >> 0x8, depth);
		Plot(screenx + 4, pal_address, pal, (data & 0x00C0) >> 0x6, depth);
		Plot(screenx + 5, pal_address, pal, (data & 0x0030) >> 0x4, depth);
		Plot(screenx + 6, pal_address, pal, (data & 0x000C) >> 0x2, depth);
		Plot(screenx + 7, pal_address, pal, (data & 0x0003) >> 0x0, depth);
	}
}

static void gfx_draw_scroll1(_u8 depth)
{
	_u8 tx, row, line;

	line = scanline + scroll1y;
	row = line & 7;	//Which row?

	//Draw Foreground scroll plane (Scroll 1)
	for (tx = 0; tx < 32; tx++)
	{
		data16 = *(_u16*)(ram + 0x9000 + ((tx + ((line >> 3) << 5)) << 1));
		
		//Draw the line of the tile
		drawPattern((tx << 3) - scroll1x, data16 & 0x01FF, 
			(data16 & 0x4000) ? 7 - row : row, data16 & 0x8000, 0x8280,
			(data16 & 0x1E00) >> 9, depth);
	}
}

static void gfx_draw_scroll2(_u8 depth)
{
	_u8 tx, row, line;

	line = scanline + scroll2y;
	row = line & 7;	//Which row?

	//Draw Background scroll plane (Scroll 2)
	for (tx = 0; tx < 32; tx++)
	{
		data16 = *(_u16*)(ram + 0x9800 + ((tx + ((line >> 3) << 5)) << 1));
		
		//Draw the line of the tile
		drawPattern((tx << 3) - scroll2x, data16 & 0x01FF, 
			(data16 & 0x4000) ? 7 - row : row, data16 & 0x8000, 0x8300,
			(data16 & 0x1E00) >> 9, depth);
	}
}

void gfx_draw_scanline_mono(void)
{
	_s16 lastSpriteX;
	_s16 lastSpriteY;
	int spr;

	//Get the current scanline
	scanline = ram[0x8009];
	cfb_offset = scanline * SCREEN_WIDTH;	//Calculate fast offset

	if (scanline == 0)
		gfx_delayed_settings();

	memset(cfb + cfb_offset, 0, SCREEN_WIDTH * sizeof(_u32));
	memset(zbuffer, 0, SCREEN_WIDTH);

	//Window colour
	data32 = *(_u16*)(_u8*)(ram + 0x83F0 + ((oowc & 7) << 1));
	r = ((data32 & 0x00F)) << 20;
	g = ((data32 & 0x0F0)) << 8;
	b = ((data32 & 0xF00)) >> 4;
	data32 = r | g | b;

	//Top
	if (scanline < winy)
	{
		for (data16 = 0; data16 < SCREEN_WIDTH; data16++)
			cfb[cfb_offset + data16] = data32;
	}
	else
	{
		//Middle
		if (scanline < winy + winh)
		{
			for (data16 = 0; data16 < min(winx, SCREEN_WIDTH); data16++)
				cfb[cfb_offset + data16] = data32;
			for (data16 = min(winx + winh, SCREEN_WIDTH); data16 < SCREEN_WIDTH; data16++)
				cfb[cfb_offset + data16] = data32;
		}
		else	//Bottom
		{
			for (data16 = 0; data16 < SCREEN_WIDTH; data16++)
				cfb[cfb_offset + data16] = data32;
		}
	}

	//Ignore above and below the window's top and bottom
	if (scanline >= winy && scanline < winy + winh)
	{
		//Background colour Enabled?
		if ((bgc & 0xC0) == 0x80)
		{
			data32 = *(_u16*)(_u8*)(ram + 0x83E0 + ((bgc & 7) << 1));
			r = ((data32 & 0x00F)) << 20;
			g = ((data32 & 0x0F0)) << 8;
			b = ((data32 & 0xF00)) >> 4;
			data32 = r | g | b;
		}
		else
			data32 = 0;
		
		//Draw background!
		for (data16 = winx; data16 < min(winx + winw, SCREEN_WIDTH); data16++)	
			cfb[data16 + cfb_offset] = data32;

		//Swap Front/Back scroll planes?
		if (planeSwap)
		{
			gfx_draw_scroll1(ZDEPTH_BACKGROUND_SCROLL);		//Swap
			gfx_draw_scroll2(ZDEPTH_FOREGROUND_SCROLL);
		}
		else
		{
			gfx_draw_scroll2(ZDEPTH_BACKGROUND_SCROLL);		//Normal
			gfx_draw_scroll1(ZDEPTH_FOREGROUND_SCROLL);
		}

		//Draw Sprites
		//Last sprite position, (defaults to top-left, sure?)
		lastSpriteX = 0;
		lastSpriteY = 0;
		for (spr = 0; spr < 64; spr++)
		{
			_u8 priority, row;
			_u8 sx = ram[0x8800 + (spr * 4) + 2];	//X position
			_u8 sy = ram[0x8800 + (spr * 4) + 3];	//Y position
			_s16 x = sx;
			_s16 y = sy;
			
			data16 = *(_u16*)(_u8*)(ram + 0x8800 + (spr * 4));
			priority = (data16 & 0x1800) >> 11;

			if (data16 & 0x0400) x = lastSpriteX + sx;	//Horizontal chain?
			if (data16 & 0x0200) y = lastSpriteY + sy;	//Vertical chain?

			//Store the position for chaining
			lastSpriteX = x;
			lastSpriteY = y;
			
			//Visible?
			if (priority == 0)	continue;

			//Scroll the sprite
			x += scrollsprx;
			y += scrollspry;

			//Off-screen?
			if (x > 248 && x < 256)	x = x - 256; else x &= 0xFF;
			if (y > 248 && y < 256)	y = y - 256; else y &= 0xFF;

			//In range?
			if (scanline >= y && scanline <= y + 7)
			{
				row = (scanline - y) & 7;	//Which row?
				drawPattern((_u8)x, data16 & 0x01FF, 
					(data16 & 0x4000) ? 7 - row : row, data16 & 0x8000,
					0x8200, ram[0x8C00 + spr] & 0xF, priority << 1); 
			}
		}
	}

	//Get delayed settings for the next scanline.
	gfx_delayed_settings();

	//==========
}

//=============================================================================
