//---------------------------------------------------------------------------
// NEOPOP : Emulator as in Dreamland
//
// Copyright (c) 2001-2002 by neopop_uk
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version. See also the license.txt file for
//	additional informations.
//---------------------------------------------------------------------------

/*
//---------------------------------------------------------------------------

  History of changes:
  ===================

20 JUL 2002 - neopop_uk
=======================================
- Cleaned and tidied up for the source release

22 JUL 2002 - neopop_uk
=======================================
- Removed setting the register file pointer to 3 in the SWI instruction
	This has fixed "Metal Slug 2" and flash saving in many games.

//---------------------------------------------------------------------------
*/

#include "neopop.h"
#include "TLCS900H_interpret.h"
#include "TLCS900H_registers.h"
#include "mem.h"
#include "interrupt.h"

//=========================================================================

/*
//=========================================================================

	42 Instructions
	===============
	0 Missing Instructions
	0 Flag warnings
	0 Incorrect cycle counts

//=========================================================================
*/

//===== NOP
void NOP()
{
	cycles = 2;
}

//===== NORMAL
void NORMAL()
{
	//Not supported
	cycles = 4;
}

//===== PUSH SR
void PUSHSR()
{
	push16(sr);
	cycles = 4;
}

//===== POP SR
void POPSR()
{
	sr = pop16();	changedSP();
	cycles = 6;
}

//===== MAX
void MAX()
{
	//Not supported
	cycles = 4;
}

//===== HALT
void HALT()
{
//	halted = true;

#ifdef NEOPOP_DEBUG
//	system_debug_message("CPU Halted.");
#endif

//	pc --;				//Stick at the halt instruction.
	cycles = 8;
}

//===== EI #3
void EI()
{
	setStatusIFF(FETCH8);
	cycles = 5;
}

//===== RETI
void RETI()
{
	_u16 temp = pop16();
	pc = pop32();
	sr = temp; changedSP();
	cycles = 12;
}

//===== LD (n), n
void LD8_8()
{
	_u8 dst = FETCH8;
	_u8 src = FETCH8;
	storeB(dst, src);
	cycles = 5;
}

//===== PUSH n
void PUSH8()
{
	_u8 data = FETCH8;
	push8(data);
	cycles = 4;
}

//===== LD (n), nn
void LD8_16()
{
	_u8 dst = FETCH8;
	_u16 src = fetch16();
	storeW(dst, src);
	cycles = 6;
}

//===== PUSH nn
void PUSH16()
{
	push16(fetch16());
	cycles = 5;
}

//===== INCF
void INCF()
{
	setStatusRFP(((sr & 0x300) >> 8) + 1);
	cycles = 2;
}

//===== DECF
void DECF()
{
	setStatusRFP(((sr & 0x300) >> 8) - 1);
	cycles = 2;
}

//===== RET condition
void RET()
{
	pc = pop32();
	cycles = 9;
}

//===== RETD dd
void RETD()
{
	_s16 d = (_s16)fetch16();
	pc = pop32();
	REGXSP += d;
	cycles = 9;
}

//===== RCF
void RCF()
{
	SETFLAG_N0;
	SETFLAG_V0;
	SETFLAG_C0;
	cycles = 2;
}

//===== SCF
void SCF()
{
	SETFLAG_H0;
	SETFLAG_N0;
	SETFLAG_C1;
	cycles = 2;
}

//===== CCF
void CCF()
{
	SETFLAG_N0;
	SETFLAG_C(!FLAG_C);
	cycles = 2;
}

//===== ZCF
void ZCF()
{
	SETFLAG_N0;
	SETFLAG_C(!FLAG_Z);
	cycles = 2;
}

//===== PUSH A
void PUSHA()
{
	push8(REGA);
	cycles = 3;
}

//===== POP A
void POPA()
{
	REGA = pop8();
	cycles = 4;
}

//===== EX F,F'
void EX()
{
	_u8 f = sr & 0xFF;
	sr = (sr & 0xFF00) | f_dash;
	f_dash = f;
	cycles = 2;
}

//===== LDF #3
void LDF()
{
	setStatusRFP(FETCH8);
	cycles = 2;
}

//===== PUSH F
void PUSHF()
{
	push8(sr & 0xFF);
	cycles = 3;
}

//===== POP F
void POPF()
{
	sr = (sr & 0xFF00) | pop8();
	cycles = 4;
}

//===== JP nn
void JP16()
{
	pc = fetch16();
	cycles = 7;
}

//===== JP nnn
void JP24()
{
	pc = fetch24();
	cycles = 7;
}

//===== CALL #16
void CALL16()
{
	_u32 target = fetch16();
	push32(pc);
	pc = target;
	cycles = 12;
}

//===== CALL #24
void CALL24()
{
	_u32 target = fetch24();
	push32(pc);
	pc = target;
	cycles = 12;
}

//===== CALR $+3+d16
void CALR()
{
	_u32 target = (_s16)fetch16() + pc;
	push32(pc);
	pc = target;
	cycles = 12;
}

//===== LD R, n
void LDB()
{
	regB(first & 7) = FETCH8;
	cycles = 2;
}

//===== PUSH RR
void PUSHW()
{
	push16(regW(first & 7));
	cycles = 3;
}

//===== LD RR, nn
void LDW()
{
	regW(first & 7) = fetch16();
	cycles = 3;
}

//===== PUSH XRR
void PUSHL()
{
	push32(regL(first & 7));
	cycles = 5;
}

//===== LD XRR, nnnn
void LDL()
{
	regL(first & 7) = fetch32();
	cycles = 5;
}

//===== POP RR
void POPW()
{
	regW(first & 7) = pop16();
	cycles = 4;
}

//===== POP XRR
void POPL()
{
	regL(first & 7) = pop32();
	cycles = 6;
}

//===== JR cc,PC + d
void JR()
{
	if (conditionCode(first & 0xF))
	{
		cycles = 8;
		pc = (_s8)FETCH8 + pc;
	}
	else
	{
		cycles = 4;
		FETCH8;
	}
}

//===== JR cc,PC + dd
void JRL()
{
	if (conditionCode(first & 0xF))
	{
		cycles = 8;
		pc = (_s16)fetch16() + pc;
	}
	else
	{
		cycles = 4;
		fetch16();
	}
}

//===== LDX dst,src
void LDX()
{
	_u8 dst, src;

	FETCH8;			//00
	dst = FETCH8;	//#8
	FETCH8;			//00
	src = FETCH8;	//#
	FETCH8;			//00

	storeB(dst, src);
	cycles = 9;
}

//===== SWI num
void SWI()
{
	cycles = 16;

	switch(first & 7)
	{
		//System Call
	case 1: push32(pc);	
			pc = loadL(0xFFFE00 + (rCodeB(0x31) << 2));
			break;

	case 3: interrupt(0);	//SWI 3
			break;

	case 4:	interrupt(1);	//SWI 4
			break;

	case 5: interrupt(2);	//SWI 5
			break;

	case 6: interrupt(3);	//SWI 6
			break;

	default:	instruction_error("SWI %d is not valid.", first & 7);
		break;
	}
}

//=============================================================================
