//============================================================================
//
//   SSSS    tt          lll  lll       
//  SS  SS   tt           ll   ll        
//  SS     tttttt  eeee   ll   ll   aaaa 
//   SSSS    tt   ee  ee  ll   ll      aa
//      SS   tt   eeeeee  ll   ll   aaaaa  --  "An Atari 2600 VCS Emulator"
//  SS  SS   tt   ee      ll   ll  aa  aa
//   SSSS     ttt  eeeee llll llll  aaaaa
//
// Copyright (c) 1995-1998 by Bradford W. Mott
//
// See the file "license" for information on usage and redistribution of
// this file, and for a DISCLAIMER OF ALL WARRANTIES.
//
// $Id: SndXBOX.cxx,v 1.1.1.1 2001/12/27 19:54:32 bwmott Exp $
//============================================================================


#include "SndXBOX.hxx"
#include <stdio.h>
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "sound.h"
#include "pcm.h"


#ifdef __cplusplus
}
#endif

// the local buffer is what the stream buffer feeds from
// note that this needs to be large enough to buffer at frameskip 11
// for 30fps games like Tapper; we will scale the value down based
// on the actual framerate of the game
#define MAX_BUFFER_SIZE			(128 * 1024)

// this is the maximum number of extra samples we will ask for
// per frame (I know this looks like a lot, but most of the
// time it will generally be nowhere close to this)
#define MAX_SAMPLE_ADJUST		16

/**
  Compute the buffer size to use based on the given sample rate

  @param The sample rate to compute the buffer size for
*/
static unsigned long computeBufferSize(int sampleRate)
{
  int t;

  for(t = 7; t <= 12; ++t)
  {
    if((1 << t) > (sampleRate / 60))
    {
      return (1 << (t - 1));
    }
  }

  return 256;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
SoundXBOX::SoundXBOX()
{
	dsound = NULL ;
	stream_buffer = NULL ;
	attenuation = 0 ;
	samples_to_read = 0 ;
}
 
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
SoundXBOX::~SoundXBOX()
{
}

int SoundXBOX::dsound_init(void)
{
	HRESULT result;


	// Only create direct sound if it hasn't already been created
	if (dsound == NULL)
	{
		// now attempt to create it
		result = DirectSoundCreate(NULL, &dsound, NULL);
		if (result != DS_OK)
		{
			goto cant_create_dsound;
		}	
	}

	current_adjustment = 0 ;

	// make a format description for what we want
	stream_format.wBitsPerSample	= 8;
	stream_format.wFormatTag		= WAVE_FORMAT_PCM;
	stream_format.nChannels			= 2;
	stream_format.nSamplesPerSec	= 44100;
	stream_format.nBlockAlign		= stream_format.wBitsPerSample * stream_format.nChannels / 8;
	stream_format.nAvgBytesPerSec	= stream_format.nSamplesPerSec * stream_format.nBlockAlign;

	// compute the buffer sizes
	stream_buffer_size = (UINT64)MAX_BUFFER_SIZE ;
//	stream_buffer_size = ((UINT64)MAX_BUFFER_SIZE * (UINT64)stream_format.nSamplesPerSec) / 44100;
	stream_buffer_size = (stream_buffer_size * stream_format.nBlockAlign) / 4;
	stream_buffer_size = (stream_buffer_size * 30) / 60 ;
	stream_buffer_size = (stream_buffer_size / 1024) * 1024;

	stream_buffer_in = 0 ;

	// compute the upper/lower thresholds
	lower_thresh = 1 * stream_buffer_size / 5;
	upper_thresh = 2 * stream_buffer_size / 5;
	

	// create the buffers
	if (dsound_create_buffers())
		goto cant_create_buffers;

	// start playing
	result = IDirectSoundBuffer_Play(stream_buffer, 0, 0, DSBPLAY_LOOPING);
	if (result != DS_OK)
	{
		goto cant_play;
	}

	return 0;
	
	// error handling
cant_play:
	dsound_destroy_buffers();
cant_create_buffers:
	IDirectSound_Release(dsound);
cant_create_dsound:
	dsound = NULL;
	return 0;
}

void SoundXBOX::dsound_destroy_buffers(void)
{
	// stop any playback
	if (stream_buffer)
		IDirectSoundBuffer_Stop(stream_buffer);

	// release the buffer
	if (stream_buffer)
		IDirectSoundBuffer_Release(stream_buffer);
	stream_buffer = NULL;
}

void SoundXBOX::init( )
{
  int sampleRate = 44100;


	samples_to_read = 2 * ( osd_start_audio_stream( 0 ) ) ; //mult by 2 for stereo

}

void SoundXBOX::copy_sample_data(unsigned char *data, int bytes_to_copy)
{	
	void *buffer1, *buffer2;
	DWORD length1, length2;
	HRESULT result;
	int cur_bytes;
	
	if ( stream_buffer == NULL )
	{
		dsound_init() ;
		if ( stream_buffer == NULL )
			return ;

	}
	// attempt to lock the stream buffer
	result = stream_buffer->Lock(stream_buffer_in, bytes_to_copy, &buffer1, &length1, &buffer2, &length2, 0);
	if (result != DS_OK)
	{
		return;
	}
	

	// adjust the input pointer
	stream_buffer_in = (stream_buffer_in + bytes_to_copy) % stream_buffer_size;
	
	// copy the first chunk
	cur_bytes = (bytes_to_copy > length1) ? length1 : bytes_to_copy;
	memcpy(buffer1, data, cur_bytes);
	
	// adjust for the number of bytes
	bytes_to_copy -= cur_bytes;
	data += cur_bytes ;
	
	// copy the second chunk
	if (bytes_to_copy != 0)
	{
		cur_bytes = (bytes_to_copy > length2) ? length2 : bytes_to_copy;
		memcpy(buffer2, data, cur_bytes);
	}
	

	result = stream_buffer->Unlock(buffer1, length1, buffer2, length2);
	if (result != DS_OK)
	{
		return;
	}
}

int SoundXBOX::dsound_create_buffers(void)
{
	HRESULT result;

	DSMIXBINVOLUMEPAIR dsmbvp[8] = {
		{DSMIXBIN_FRONT_LEFT, DSBVOLUME_MAX},   // left channel
		{DSMIXBIN_FRONT_RIGHT, DSBVOLUME_MAX},  // right channel
		{DSMIXBIN_FRONT_CENTER, DSBVOLUME_MAX}, // left channel
		{DSMIXBIN_FRONT_CENTER, DSBVOLUME_MAX}, // right channel
		{DSMIXBIN_BACK_LEFT, DSBVOLUME_MAX},    // left channel
		{DSMIXBIN_BACK_RIGHT, DSBVOLUME_MAX},   // right channel
		{DSMIXBIN_LOW_FREQUENCY, DSBVOLUME_MAX},    // left channel
		{DSMIXBIN_LOW_FREQUENCY, DSBVOLUME_MAX}};   // right channel
		
	DSMIXBINS dsmb;
	dsmb.dwMixBinCount = 8;
	dsmb.lpMixBinVolumePairs = dsmbvp;
	
	// create a buffer desc for the stream buffer
	stream_desc.dwSize				= sizeof(stream_desc);
	stream_desc.dwFlags				= 0;
	stream_desc.dwBufferBytes 		= stream_buffer_size;
	stream_desc.lpwfxFormat			= &stream_format;
	stream_desc.lpMixBins			= &dsmb;
	
	//	stream_desc.fred		= DSMIXBIN_FRONT_LEFT | DSMIXBIN_FRONT_RIGHT;
	
	// If our stream buffer hasn't been freed, do so now
	if (stream_buffer != NULL)
	{
		dsound_destroy_buffers();
	}
	
	// create the stream buffer
	if ((result = dsound->CreateSoundBuffer(&stream_desc, &stream_buffer, NULL)) != DS_OK)
	{
		goto cant_create_buffer;
	}
	
	DWORD locked;
	void *buffer;
	
	// lock the buffer
	result = IDirectSoundBuffer_Lock(stream_buffer, 0, stream_buffer_size, &buffer, &locked, NULL, NULL, 0);
	if (result != DS_OK)
	{
		fprintf(stderr, "Error locking stream buffer: %08x\n", (UINT32)result);
		goto cant_lock_buffer;
	}

	// clear the buffer to remove any noise in it
	memset(buffer, 0, locked);
	
	return 0;
	
cant_lock_buffer:
	stream_buffer->Release();
cant_create_buffer:
	stream_buffer = NULL;
	
	return 0;
}

int SoundXBOX::osd_start_audio_stream(int stereo)
{

	// attempt to initialize directsound
	if (dsound_init())
		return 1;

	// set the startup volume
	osd_set_mastervolume(attenuation);

	// determine the number of samples per frame
	samples_per_frame = (double) 44100 / (double)60;

	// compute how many samples to generate the first frame
	samples_left_over = samples_per_frame;
	samples_this_frame = (UINT32)samples_left_over;
	samples_left_over -= (double)samples_this_frame;

	// return the samples to play the first frame
	return samples_this_frame;
}

int SoundXBOX::bytes_in_stream_buffer(void)
{
	DWORD play_position, write_position;
	HRESULT result;

	result = IDirectSoundBuffer_GetCurrentPosition(stream_buffer, &play_position, &write_position);
	if (stream_buffer_in > play_position)
		return stream_buffer_in - play_position;
	else
		return stream_buffer_size + stream_buffer_in - play_position;
}

void SoundXBOX::update_sample_adjustment(int buffered)
{
	static int consecutive_lows = 0;
	static int consecutive_mids = 0;
	static int consecutive_highs = 0;

	// if we're not throttled don't bother
	if ( 1 )
	{
		consecutive_lows = 0;
		consecutive_mids = 0;
		consecutive_highs = 0;
		current_adjustment = 0;
		return;
	}

}

int SoundXBOX::osd_update_audio_stream(unsigned char *buffer)
{
	// if nothing to do, don't do it
	if ( stream_buffer)
	{
//		int original_bytes = bytes_in_stream_buffer();
		int input_bytes = samples_this_frame*2 ;
		//int input_bytes = samples_this_frame * stream_format.nBlockAlign;
		int final_bytes;

		// update the sample adjustment
		//update_sample_adjustment(original_bytes);

		// copy data into the sound buffer
		copy_sample_data(buffer, input_bytes);

		// check for overflows
		//final_bytes = bytes_in_stream_buffer();
	}


	// compute how many samples to generate next frame
	samples_left_over += samples_per_frame;
	samples_this_frame = (UINT32)samples_left_over;
	samples_left_over -= (double)samples_this_frame;

	//samples_this_frame += current_adjustment;

	// return the samples to play this next frame
	return samples_this_frame;
}

void SoundXBOX::osd_set_mastervolume(int _attenuation)
{
	// clamp the attenuation to 0-32 range
	if (_attenuation > 0)
		_attenuation = 0;
	if (_attenuation < -32)
		_attenuation = -32;
	attenuation = _attenuation;

	// set the master volume
	if (stream_buffer)
		IDirectSoundBuffer_SetVolume(stream_buffer, attenuation * 100);
}

//============================================================
//	osd_stop_audio_stream
//============================================================

void SoundXBOX::osd_stop_audio_stream(void)
{

	// kill the buffers and dsound
	dsound_destroy_buffers();
	//dsound_kill();

}


void SoundXBOX::cleanup()
{
	osd_stop_audio_stream() ;
}

void SoundXBOX::process()
{
	unsigned char *sndbuf ;
	int            num_written ;

	num_written = 0 ;
	sndbuf = (unsigned char*)malloc( 5*1024 ) ;

	sound_mix();
	memcpy( sndbuf, pcm.buf, pcm.pos ) ;
	
	
	/*
	while ( num_written < 1470 )
	{
		if ( pcm.pos == 0 )
		{
			sound_mix();
		}

		memcpy( sndbuf + num_written, pcm.buf, pcm.pos ) ;
		num_written += pcm.pos ;
		pcm.pos = 0 ;
	}*/
/*
	if ( num_written > 1470 )
	{
		pcm.pos = num_written - 1470 ;
		while ( pcm.pos >= pcm.len )
			pcm.pos -= pcm.len ;

		memcpy( pcm.buf, sndbuf + 1470, pcm.pos ) ;
	}*/
//	samples_to_read = 2* ( osd_update_audio_stream( sndbuf ) );  // mult by 2 for stereo
	copy_sample_data(sndbuf, pcm.pos);
	pcm.pos = 0 ;

	free(sndbuf) ;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void SoundXBOX::mute(bool state)
{
}

